// Copyright (C) 2012-2014 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

/**
 * @file io_event.hh
 * @author Kornel Kisielewicz
 * @brief 
 */

#ifndef NV_CORE_IO_EVENT_HH
#define NV_CORE_IO_EVENT_HH

#include <nv/core/common.hh>

namespace nv
{

	// Generate the key_code enum
	enum key_code
	{
#	define NV_KEY( id, val ) id = val,
#		include <nv/detail/key_list.inc>
#	undef NV_KEY
	};
	
	// Generate the mouse_code enum
	enum mouse_code
	{
#	define NV_MOUSE( id, val ) id = val,
#		include <nv/detail/mouse_list.inc>
#	undef NV_MOUSE
	};

	// Generate the io_event_code enum
	enum io_event_code
	{
#	define NV_IO_EVENT( id ) id,
#		include <nv/detail/io_event_list.inc>
#	undef NV_IO_EVENT
	};

	struct key_event
	{
		/// Input event ASCII code
		char8 ascii;

		/// Input event local code
		key_code code;

		/// True if shift key is present
		bool shift;

		/// True if control key is present
		bool control;

		/// True if alt key is present
		bool alt;

		/// True if pressed
		bool pressed;
	};

	struct mouse_button_event
	{
		/// X position where mouse was clicked.
		uint16 x;
		/// Y position where mouse was clicked.
		uint16 y;
		/// Button that was clicked.
		uint32 button;
		/// True if pressed
		bool pressed;
		/// Mouse button code
		mouse_code code;
	};

	struct mouse_wheel_event
	{
		/// amount scrolled horizontally positive to the right
		sint32 x;
		/// amount scrolled vertically 
		sint32 y;
	};

	struct mouse_move_event
	{
		/// X Position the mouse moved to.
		uint16 x;
		/// Y Position the mouse moved to.
		uint16 y;
		/// Distance in x direction mouse was moved.
		sint16 rx;
		/// Distance in y direction mouse was moved.
		sint16 ry;
		/// True if pressed
		bool pressed;
		/// Mouse button code
		mouse_code code;
	};

	struct joy_button_event
	{
		/// Joystick ID
		sint32 id;
		/// Button that is affected
		uint8 button;
		/// True if pressed
		bool pressed;
	};

	struct joy_axis_event
	{
		/// Joystick ID
		sint32 id;
		/// Axis ID
		uint8 axis;
		/// Value
		sint16 value;
	};

	struct joy_hat_event
	{
		/// Joystick ID
		sint32 id;
		/// Hat ID
		uint8 hat;
		/// Value
		sint16 value;
	};

	struct joy_ball_event
	{
		/// Joystick ID
		sint32 id;
		/// Ball ID
		uint8 ball;
		/// Relative X
		sint16 rx;
		/// Relative Y
		sint16 ry;
	};

	struct resize_event
	{
		/// New x size of the object.
		sint32 x;
		/// New y size of the object.
		sint32 y;
	};

	struct active_event
	{
		/// Whether focus was gained or lost.
		bool gain;
	};

	struct system_event
	{
		uint8  sys_type;
		uint32 param1;
		uint32 param2;
	};

	struct io_event
	{
		io_event_code type;
		union
		{
			key_event          key;
			mouse_button_event mbutton;
			mouse_move_event   mmove;
			mouse_wheel_event  mwheel;
			joy_button_event   jbutton;
			joy_axis_event     jaxis;
			joy_hat_event      jhat;
			joy_ball_event     jball;
			resize_event       resize;
			active_event       active;
			system_event       system;
		};
	};

	/**
	 * Gets the name of the key given the code.
	 *
	 * @param key The code value of the key.
	 * @returns The name of the key.
	 */
	const char* get_key_name( key_code key );

	/**
	 * Gets the name of the mouse button given the code.
	 *
	 * @param button The code value of the mouse button.
	 * @returns The name of the button.
	 */
	const char* get_mouse_name( mouse_code button );

	/**
	 * Gets the name of the IO event given the code.
	 *
	 * @param event The code value of the IO event.
	 * @returns The name of the event.
	 */
	const char* get_io_event_name( io_event_code event );

	/**
	 * Registers all events to the specified database.
	 *
	 * @param db The database to store all event data in.
	 */
	//void register_io_types( type_database* db );
}

#endif // NV_CORE_IO_EVENT_HH
