// Copyright (C) 2012-2014 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

/**
 * @file render_state.hh
 * @author Kornel Kisielewicz epyon@chaosforge.org
 * @brief Render state class
 */

#ifndef NV_RENDER_STATE_HH
#define NV_RENDER_STATE_HH

#include <nv/core/common.hh>
#include <nv/core/math.hh>
#include <nv/interface/clear_state.hh>
#include <nv/core/string.hh>

namespace nv
{
	struct depth_test
	{
		enum function_type
		{
			NEVER,
			LESS,
			EQUAL,
			LESS_OR_EQUAL,
			GREATER,
			NOT_EQUAL,
			GREATER_OR_EQUAL,
			ALWAYS
		};
		
		bool enabled;
		function_type function;

		depth_test() : enabled( true ), function( LESS ) {}
	};

	struct polygon_mode
	{
		enum fill_type
		{
			POINT,
			LINE,
			FILL
		};

		fill_type fill;

		polygon_mode() : fill( FILL ) {}
	};

	struct blending
	{
		enum factor
		{
			ZERO,
			ONE,
			SRC_COLOR,
			SRC_ALPHA,
			DST_COLOR,
			DST_ALPHA,
			CONSTANT_COLOR,
			CONSTANT_ALPHA,
			ONE_MINUS_SRC_COLOR,
			ONE_MINUS_SRC_ALPHA,
			ONE_MINUS_DST_COLOR,
			ONE_MINUS_DST_ALPHA,
			ONE_MINUS_CONSTANT_COLOR,
			ONE_MINUS_CONSTANT_ALPHA,
			SRC_ALPHA_SATURATE,
		};

		enum equation
		{
			ADD,
			MINIMUM,
			MAXIMUM,
			SUBTRACT,
			REVERSE_SUBTRACT,
		};

		bool enabled;
		factor src_rgb_factor;
		factor src_alpha_factor;
		factor dst_rgb_factor;
		factor dst_alpha_factor;
		equation rgb_equation;
		equation alpha_equation;
		vec4 color;

		blending() :
			enabled( false ),
			src_rgb_factor( ONE ), src_alpha_factor( ONE ), 
			dst_rgb_factor( ZERO ), dst_alpha_factor( ZERO ), 
			rgb_equation( ADD ), alpha_equation( ADD ),
			color( vec4() ) {}
	};

	struct stencil_test_face
	{
		enum operation
		{
			ZERO,
			INVERT,
			KEEP,
			REPLACE,
			INCREMENT,
			DECREMENT,
			INCREMENT_WRAP,
			DECREMENT_WRAP
		};

		enum function_type
		{
			NEVER,
			LESS,
			EQUAL,
			LESS_OR_EQUAL,
			GREATER,
			NOT_EQUAL,
			GREATER_OR_EQUAL,
			ALWAYS
		};

		operation op_fail;
		operation op_depth_pass;
		operation op_depth_fail;

		function_type function;
		int ref_value;
		uint32 mask;

		stencil_test_face()	:
			op_fail( KEEP ),
			op_depth_pass( KEEP ),
			op_depth_fail( KEEP ),
			function( ALWAYS ),
			ref_value( 0 ),
			mask( uint32(-1) )
		{}
	};

	struct stencil_test
	{
		bool enabled;
		stencil_test_face front_face;
		stencil_test_face back_face;

		stencil_test() :
			enabled( false ),
			front_face(),
			back_face()
		{}
	};
	
	struct culling
	{
		enum face_type
		{
			FRONT,
			BACK,
			FRONT_AND_BACK
		};

		enum order_type
		{
			CW,
			CCW
		};

		bool enabled;
		face_type face;
		order_type order;

		culling() : enabled( true ), face( BACK ), order( CCW ) {}
	};

	struct depth_range
	{
		f64 near;
		f64 far;

		depth_range() : near(0.0), far(1.0) {}
	};

	struct render_state
	{
		nv::stencil_test stencil_test;
		nv::depth_test   depth_test;
		nv::scissor_test scissor_test;
		nv::depth_range  depth_range;
		nv::blending     blending;
		nv::culling      culling;
		nv::color_mask   color_mask;
		nv::polygon_mode polygon_mode;
		bool depth_mask;
		render_state() : depth_mask( true ) {}
	};

} // namespace nv


#endif // NV_RENDER_STATE_HH
