// Copyright (C) 2012-2014 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh
//
// TODO: decouple from the type database
// TODO: paths for function registration?


#ifndef NV_LUA_STATE_HH
#define NV_LUA_STATE_HH

#include <nv/core/common.hh>
#include <nv/core/flags.hh>
#include <istream>
#include <map>

#include <nv/lua/lua_path.hh>
#include <nv/lua/lua_values.hh>
#include <nv/lua/lua_dispatch.hh>
#include <string>

namespace nv
{
	namespace lua
	{

		const int ret_multi = -1;

		class state_wrapper
		{
		public:
			state_wrapper( lua_State* a_state, bool a_owner ) : m_state( a_state ), m_owner( a_owner ), m_global( true ) {}
			virtual ~state_wrapper();

			template < typename R >
			R call( const path& p )
			{
				if ( push_function( p, m_global ) )
				{
					if ( call_function( 0, detail::return_count< R >::value ) == 0 )
					{
						return detail::pop_return_value<R>( m_state );
					}
				}
				return R();
			}
			template < typename R, typename T1 >
			R call( const path& p, const T1& p1 )
			{
				if ( push_function( p, m_global ) )
				{
					detail::push_value( m_state, p1 );
					if ( call_function( 1, detail::return_count< R >::value ) == 0 )
					{
						return detail::pop_return_value<R>( m_state );
					}
				}
				return R();
			}
			template < typename R, typename T1, typename T2 >
			R call( const path& p, const T1& p1, const T2& p2 )
			{
				if ( push_function( p, m_global ) )
				{
					detail::push_values( m_state, p1, p2 );
					if ( call_function( 2, detail::return_count< R >::value ) == 0 )
					{
						return detail::pop_return_value<R>( m_state );
					}
				}
				return R();
			}
			template < typename R, typename T1, typename T2, typename T3 >
			R call( const path& p, const T1& p1, const T2& p2, const T3& p3 )
			{
				if ( push_function( p, m_global ) )
				{
					detail::push_values( m_state, p1, p2, p3 );
					if ( call_function( 3, detail::return_count< R >::value ) == 0 )
					{
						return detail::pop_return_value<R>( m_state );
					}
				}
				return R();
			}
			template < typename R, typename T1, typename T2, typename T3, typename T4 >
			R call( const path& p, const T1& p1, const T2& p2, const T3& p3, const T4& p4 )
			{
				if ( push_function( p, m_global ) )
				{
					detail::push_value( m_state, p1, p2, p3, p4 );
					if ( call_function( 4, detail::return_count< R >::value ) == 0 )
					{
						return detail::pop_return_value<R>( m_state );
					}
				}
				return R();
			}

			template< typename R >
			R get( const path& p )
			{
				if ( p.resolve( m_state, m_global ) )
				{
					return detail::pop_return_value<R>( m_state );
				}
				return R();
			}

			template< typename R, typename T >
			R get( const T& key )
			{
				if ( m_global ) push_global_table();
				detail::push_value( m_state, key );
				call_get();
				if ( m_global ) pop_global_table();
				return detail::pop_return_value<R>( m_state );
			}

			template< typename R, typename T >
			R raw_get( const T& key )
			{
				if ( m_global ) push_global_table();
				detail::push_value( m_state, key );
				call_get_raw();
				if ( m_global ) pop_global_table();
				return detail::pop_return_value<R>( m_state );
			}

			template< typename R >
			R get( const path& p, const R& def )
			{
				if ( p.resolve( m_state, m_global ) )
				{
					return detail::pop_return_value<R>( m_state, def );
				}
				return R();
			}

			template< typename R, typename T >
			R get( const T& key, const R& def )
			{
				if ( m_global ) push_global_table();
				detail::push_value( m_state, key );
				call_get();
				if ( m_global ) pop_global_table();
				return detail::pop_return_value<R>( m_state, def );
			}

			template< typename R, typename T >
			R raw_get( const T& key, const R& def )
			{
				if ( m_global ) push_global_table();
				detail::push_value( m_state, key );
				call_get_raw();
				if ( m_global ) pop_global_table();
				return detail::pop_return_value<R>( m_state, def );
			}
			bool is_defined( const path& p ) { return is_defined( p, m_global ); }
			void register_native_function( lfunction f, const char* name );

			template < typename F, F f >
			void register_function( const char* name )
			{
				register_native_function( detail::function_wrapper< F, f >, name );
			}

			template < typename C, typename F, F f >
			struct register_member
			{
				register_member( const char* ) { }
			};

			template < typename C, typename F, F f >
			void register_function( const char* name )
			{
				register_native_function( detail::object_method_wrapper< C, F, f >, name );
			}

		protected:
			bool is_defined( const path& p, bool global );
			bool push_function( const path& p, bool global );
			int call_function( int nargs, int nresults );
		private:
			void push_global_table();
			void pop_global_table();
			void call_get();
			void call_get_raw();
		protected:
			lua_State* m_state;
			bool m_owner;
			bool m_global;
		};

		class stack_guard
		{
		public:
			stack_guard( state* aL );
			stack_guard( state& aL );
			int get_level() const { return m_level; }
			~stack_guard();
		private:
			state* L;
			int m_level;
		};

		class table_guard;
		class state : public state_wrapper
		{
			friend class stack_guard;
			friend class table_guard;

		public:
			explicit state( bool load_libs = false );
			explicit state( lua_State* state );
			bool do_string( const std::string& code, const std::string& name, int rvalues = 0 );
			bool do_stream( std::istream& stream, const std::string& name );
			bool do_file( const std::string& filename );
			int get_stack_size();
			void log_stack();
			lua_State* get_raw();
			ref register_object( void* o, const char* lua_name );
			ref register_proto( const char* id, const char* storage );
			void store_metadata( ref object_index, const std::string& metaname, void* pointer );
			void unregister_object( ref object_index );

			void register_enum( const char* name, int value );

			void register_native_object_method( const char* lua_name, const char* name, lfunction f );
			template < typename F, F f >
			void register_object_method( const char* lua_name, const char* name )
			{
				register_native_object_method( lua_name, name, detail::object_method_wrapper< typename memfn_class_type<F>::type, F, f > );
			}
			operator lua_State*() { return m_state; }
		private:
			int load_string( const std::string& code, const std::string& name );
			int load_stream( std::istream& stream, const std::string& name );
			int load_file( const std::string& filename );
			int do_current( const std::string& name, int rvalues = 0 );
			void deep_pointer_copy( int index, void* obj );
		};

		class table_guard : public state_wrapper
		{
		public:
			table_guard( state* lstate, const path& p, bool global = true );
			table_guard( const table_guard& parent, const path& p );
			virtual ~table_guard();
			size_t get_size();
			bool has_field( const std::string& element );
			std::string get_string( const std::string& element, const std::string& defval = "" );
			char get_char( const std::string& element, char defval = ' ' );
			int get_integer( const std::string& element, int defval = 0 );
			unsigned get_unsigned( const std::string& element, unsigned defval = 0 );
			double get_double( const std::string& element, double defval = 0.0 );
			float get_float( const std::string& element, float defval = 0.0 );
			bool get_boolean( const std::string& element, bool defval = false );
			bool is_table( const std::string& element );
			bool is_number( const std::string& element );
			bool is_boolean( const std::string& element );
			bool is_string( const std::string& element );
		private:
			int m_level;
		};



	} // namespace lua

} // namespace nv

#endif // NV_LUA_STATE_HH
