// Copyright (C) 2011-2014 ChaosForge Ltd
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

#include "nv/gfx/image.hh"

#include <algorithm>

using namespace nv;

image::image( glm::ivec2 size, size_t depth )
	: m_size( size ), m_depth( depth ), m_data( nullptr )
{
	m_data = new uint8[ static_cast<uint32>( m_size.x * m_size.y ) * m_depth ];
}

image::image( image_data* data )
	: m_size( data->get_size() ), m_depth( data->get_depth() ), m_data( data->release_data() )
{
	NV_ASSERT( m_data, "image created from empty image_data!" );
}


image::image( glm::ivec2 size, size_t depth, const uint8 * data, bool reversed )
	: m_size( size ), m_depth( depth ), m_data( nullptr )
{
	sint32 bsize = m_size.x * m_size.y * static_cast<sint32>( m_depth );
	m_data = new uint8[ bsize ];

	if ( reversed )
	{
		sint32 bline = m_size.x * static_cast<sint32>( m_depth );
		for( int i = 0; i < m_size.y; ++i )
		{
			std::copy( data + i * bline, data + (i + 1) * bline, m_data + bsize - ( i + 1 ) * bline );
		}

	}
	else 
	{
		std::copy( data, data + bsize, m_data );
	}
}

void image::fill( uint8 value )
{
	std::fill( m_data, m_data + m_size.x * m_size.y * (int)m_depth, value );
}

void image::fill( region r, uint8 value, int stride )
{
	if ( stride == 0 ) stride = r.size.x * static_cast<sint32>( m_depth );

	sint32 bpos  = (r.pos.y*m_size.x + r.pos.x ) * static_cast<sint32>( m_depth );
	sint32 bline = m_size.x*static_cast<sint32>( m_depth );

	for( int i = 0; i < r.size.y; ++i )
	{
		// TODO: test 
		std::fill( m_data + bpos + bline * i, m_data + bpos + bline * i + stride, value );
	}
}


void image::set_region( region r, const uint8 * data, int stride )
{
	if ( stride == 0 ) stride = r.size.x * static_cast<sint32>( m_depth );
	
	sint32 bpos  = (r.pos.y*m_size.x + r.pos.x ) * static_cast<sint32>( m_depth );
	sint32 bline = m_size.x*static_cast<sint32>( m_depth );

	for( int i = 0; i < r.size.y; ++i )
 	{
// TODO: test if same as old:
//		memcpy( m_data+((r.pos.y+i)*m_size.x + r.pos.x ) * m_depth, 
//			data + (i*stride), r.size.x * m_depth );
		std::copy( data + i*stride, data + (i+1)*stride, m_data + bpos + bline * i );
	}
}

void image::set_region( region r, const image_data* idata )
{
	if ( idata->get_depth() == m_depth )
	{
		set_region( r, idata->get_data() );
		return;
	}

	fill( r, 255 );

	uint32 bpos       = static_cast< uint32 >( r.pos.y*m_size.x + r.pos.x ) * m_depth;
	uint32 bline      = static_cast< uint32 >( m_size.x ) * m_depth;

	uint32 rsizex     = static_cast< uint32 >( r.size.x );
	uint32 rsizey     = static_cast< uint32 >( r.size.y );
	const uint8* data = idata->get_data();
	size_t depth      = idata->get_depth();
	uint32 dstride    = rsizex * depth;

	for( uint32 y = 0; y < rsizey; ++y )
	{
		uint32 pos = bpos + bline * y;
		for( uint32 x = 0; x < rsizex; ++x )
		{
			uint32 xy = pos + x * m_depth;
			for( size_t e = 0; e < depth; ++e )
			{
				m_data[ xy + e ] = data[ y*dstride + x * depth + e ];
			}
		}
	}
}


image::~image()
{
	delete[] m_data;
}


