#include <nv/gl/gl_device.hh>
#include <nv/gfx/image.hh>
#include <nv/interface/context.hh>
#include <nv/interface/window.hh>
#include <nv/core/logging.hh>
#include <nv/core/logger.hh>
#include <nv/core/string.hh>
#include <nv/core/random.hh>
#include <nv/gfx/sliced_buffer.hh>

#define INDEXED_TEST

struct vertex
{
	nv::ivec2 coord;
	nv::vec4  color;
	vertex() {}
	vertex( const nv::ivec2& coord, const nv::vec4& color )
		: coord( coord ), color( color ) {}
};

#ifndef INDEXED_TEST
struct quad
{
	vertex vtx[6];
	quad( const nv::ivec2& coorda, const nv::ivec2& coordb, const nv::vec4& color )
	{
		vtx[0].color = color;
		vtx[1].color = color;
		vtx[2].color = color;
		vtx[3].color = color;
		vtx[4].color = color;
		vtx[5].color = color;
		vtx[0].coord = coorda;
		vtx[1].coord = nv::ivec2( coorda.x, coordb.y );
		vtx[2].coord = coordb;
		vtx[3].coord = coordb;
		vtx[4].coord = nv::ivec2( coordb.x, coorda.y );
		vtx[5].coord = coorda;
	}
	quad( const nv::ivec2& coorda, const nv::ivec2& coordb, const nv::ivec2& coordc, const nv::ivec2& coordd, const nv::vec4& color )
	{
		vtx[0].color = color;
		vtx[1].color = color;
		vtx[2].color = color;
		vtx[3].color = color;
		vtx[4].color = color;
		vtx[5].color = color;
		vtx[0].coord = coorda;
		vtx[1].coord = coordb;
		vtx[2].coord = coordc;
		vtx[3].coord = coordc;
		vtx[4].coord = coordd;
		vtx[5].coord = coorda;
	}

};
#endif 

#ifdef INDEXED_TEST
typedef nv::indexed_sliced_buffer<vertex> gcache;
typedef nv::indexed_buffer_slice<vertex> gslice;
#else
typedef nv::sliced_buffer<quad> gcache;
typedef nv::buffer_slice<quad> gslice;
#endif

struct app_window
{
	app_window( gcache* cache, const glm::ivec2& a, const glm::ivec2& b, const glm::vec4& color )
		: m_slice( cache ), m_simple( false ), m_a( a ), m_b( b ), m_c( color )
	{
		create_complex();
	}

	void change_color( const glm::vec4& color )
	{
		m_c = color;
		glm::vec4 dcolor( color.x * 0.5, color.y * 0.5, color.z * 0.5, 1.0 );
		#ifdef INDEXED_TEST
		std::vector<vertex>& v = m_slice.lock_vertices();
		size_t size   = v.size();
		size_t dcount = 8;
		size_t dmin   = 8;
		size_t count  = size;
		#else
		std::vector<quad>& v = m_slice.lock();
		size_t size   = v.size();
		size_t dcount = (size - 1) * 6;
		size_t dmin   = 1;
		size_t count  = size * 6;
		#endif
		vertex* vtx = (vertex*)v.data();
		if ( size > dmin )
		{
			for (size_t i = 0; i < dcount; ++i )
				vtx[i].color = dcolor;
			for (size_t i = dcount; i < count; ++i )
				vtx[i].color = color;
		}
		else
		{
			for (size_t i = 0; i < count; ++i )
				vtx[i].color = color;
		}
	}

	void simplify_toggle()
	{
		if ( !m_simple )
		{
			NV_LOG( nv::LOG_INFO, "Simplifing" );
			create_simple();
		}
		else
		{
			NV_LOG( nv::LOG_INFO, "Complexifing" );
			create_complex();
		}
	}

	void create_simple()
	{
		#ifdef INDEXED_TEST
		std::vector<vertex>& v     = m_slice.lock_vertices();
		std::vector<nv::uint16>& i = m_slice.lock_indices();
		v.clear();
		i.clear();
		v.emplace_back( m_a, m_c );
		v.emplace_back( nv::ivec2( m_a.x, m_b.y ), m_c );
		v.emplace_back( m_b, m_c );
		v.emplace_back( nv::ivec2( m_b.x, m_a.y ), m_c );
		nv::uint16 tmp[] = { 0, 1, 2, 2, 3, 0 };
		i.insert( i.end(), tmp, std::end( tmp ) );
		#else
		std::vector<quad>& v = m_slice.lock();
		v.clear();
		v.emplace_back( m_a, m_b, m_c );
		#endif
		m_simple = true;
	}

	void create_complex()
	{
		glm::vec4 dcolor( m_c.x * 0.5, m_c.y * 0.5, m_c.z * 0.5, 1.0 );
		nv::ivec2 a2( m_a.x, m_b.y );
		nv::ivec2 b2( m_b.x, m_a.y );
		nv::ivec2 t1( 10, 10 );
		nv::ivec2 t2( -10, 10 );
		#ifdef INDEXED_TEST
		std::vector<vertex>& v     = m_slice.lock_vertices();
		std::vector<nv::uint16>& i = m_slice.lock_indices();
		v.clear();
		i.clear();
		v.emplace_back( m_a- t1, dcolor ); // 0
		v.emplace_back( a2 + t2, dcolor ); // 1
		v.emplace_back( m_b+ t1, dcolor ); // 2
		v.emplace_back( b2 - t2, dcolor ); // 3

		v.emplace_back( m_a, dcolor ); // 4
		v.emplace_back( a2, dcolor );  // 5
		v.emplace_back( m_b, dcolor ); // 6
		v.emplace_back( b2, dcolor );  // 7
		nv::uint16 tmp[] = { 
			0, 4, 7, 7, 3, 0,
			0, 1, 5, 5, 4, 0,
			1, 2, 6, 6, 5, 1,
			2, 3, 7, 7, 6, 2,
		};
		i.insert( i.end(), tmp, std::end( tmp ) );

		v.emplace_back( m_a, m_c );    // 8
		v.emplace_back( a2, m_c );     // 9
		v.emplace_back( m_b, m_c );    // 10
		v.emplace_back( b2, m_c );     // 11
		nv::uint16 tmp2[] = { 8, 9, 10, 10, 11, 8 };
		i.insert( i.end(), tmp2, std::end( tmp2 ) );

		#else
		std::vector<quad>& v = m_slice.lock();
		v.clear();
		v.emplace_back( m_a - t1, m_a,       b2, b2 - t2, dcolor );
		v.emplace_back( m_a - t1, a2 + t2, a2, m_a,       dcolor );
		v.emplace_back( a2 + t2, m_b + t1, m_b, a2, dcolor );
		v.emplace_back( m_b + t1, b2 - t2, b2, m_b, dcolor );
		v.emplace_back( m_a, m_b, m_c );
		#endif
		m_simple = false;
	}

	void draw()
	{
		m_slice.commit();
	}

	gslice m_slice;
	bool m_simple;
	nv::ivec2 m_a;
	nv::ivec2 m_b;
	nv::vec4 m_c;
};

class application
{
public:
	application();
	bool initialize();
	bool run();
	void kill_window();
	void spawn_window();
	void simplify_window();
	void recolor_window();
	~application();
protected:
	nv::device*  m_device;
	nv::context* m_context;
	nv::window*  m_window;
	nv::clear_state m_clear_state;
	nv::render_state m_render_state;
	
	gcache* m_quad_cache;
	std::vector<app_window>  m_windows;

	nv::program       m_program;
	nv::vertex_array  m_va;
	unsigned int m_count;
};

application::application()
{
	m_device  = new nv::gl_device();
	m_window  = m_device->create_window( 800, 600, false );
	m_context = m_window->get_context();

	m_clear_state.buffers = nv::clear_state::COLOR_AND_DEPTH_BUFFER;
	m_render_state.depth_test.enabled = false;
	m_render_state.culling.enabled    = false;
	m_render_state.blending.enabled   = false;
	m_render_state.blending.src_rgb_factor   = nv::blending::SRC_ALPHA;
	m_render_state.blending.dst_rgb_factor   = nv::blending::ONE_MINUS_SRC_ALPHA;
	m_render_state.blending.src_alpha_factor = nv::blending::SRC_ALPHA;
	m_render_state.blending.dst_alpha_factor = nv::blending::ONE_MINUS_SRC_ALPHA;
}

bool application::initialize()
{
	{ 
		m_program   = m_device->create_program( nv::slurp( "cachebuf.vert" ), nv::slurp( "cachebuf.frag" ) );
		m_va        = m_context->create_vertex_array();

		#ifdef INDEXED_TEST
		m_quad_cache = new gcache( m_context, nv::DYNAMIC_DRAW, 200, 200 );
		m_context->set_index_buffer( m_va, m_quad_cache->get_index_buffer(), nv::USHORT, false );
		nv::buffer buffer = m_quad_cache->get_vertex_buffer();
		#else
		m_quad_cache = new gcache( m_context, nv::VERTEX_BUFFER, nv::DYNAMIC_DRAW, 20 );
		nv::buffer buffer = m_quad_cache->get_buffer();
		#endif

		m_context->add_vertex_buffer( m_va, nv::slot::POSITION, buffer, nv::INT,   2, 0, sizeof( vertex ), false );
		m_context->add_vertex_buffer( m_va, nv::slot::COLOR,    buffer, nv::FLOAT, 4, offset_of( &vertex::color ), sizeof( vertex ), false );
	}
	return true;
}

bool application::run()
{
	int keypress = 0;
	glm::mat4 projection = glm::ortho( 0.0f, 800.0f, 600.0f, 0.0f, -1.0f, 1.0f );
	m_device->set_uniform( m_program, "nv_projection", glm::mat4(projection) );

	while(!keypress) 
	{
		for ( auto& w : m_windows )
		{
			w.draw();
		}
		if (m_quad_cache->commit() )
		{
			#ifdef INDEXED_TEST
			m_context->set_index_buffer( m_va, m_quad_cache->get_index_buffer(), nv::USHORT, false );
			nv::buffer buffer = m_quad_cache->get_vertex_buffer();
			m_context->replace_vertex_buffer( m_va, buffer, nv::slot::POSITION, false );
			#else
			nv::buffer buffer = m_quad_cache->get_buffer();
			m_context->replace_vertex_buffer( m_va, buffer, false );
			#endif
		}

		m_window->get_context()->clear( m_clear_state );
		#ifdef INDEXED_TEST
		size_t draw_size = m_quad_cache->get_index_size();
		#else
		size_t draw_size = m_quad_cache->get_size() * 6;
		#endif
		m_window->get_context()->draw( nv::TRIANGLES, m_render_state, m_program, m_va, draw_size );
		m_window->swap_buffers();

		nv::io_event event;
		while(m_window->poll_event(event)) 
		{      
			switch (event.type) 
			{
			case nv::EV_QUIT:
				keypress = 1;
				break;
			case nv::EV_KEY:
				if (event.key.pressed)
				{
					switch (event.key.code) 
					{
					case nv::KEY_N      : spawn_window(); break;
					case nv::KEY_S      : simplify_window(); break;
					case nv::KEY_C      : recolor_window(); break;
					case nv::KEY_K      : kill_window(); break;
					case nv::KEY_ESCAPE : keypress = 1; break;
					}
				}
				break;
			}
		}
	}
	return true;
}

void application::spawn_window()
{
	glm::ivec2 a = nv::random::get().range( nv::ivec2(), nv::ivec2( 600, 400 ) );
	glm::ivec2 b = nv::random::get().range( nv::ivec2(), nv::ivec2( 200, 200 ) );
	NV_LOG( nv::LOG_INFO, "Spawn (" << a.x << "," << a.y << "x" << b.x << "," << b.y << ")" );
	m_windows.emplace_back( m_quad_cache, a, a + b, glm::vec4( 0, 0, 1, 1 ) );
}

void application::kill_window()
{
	if ( m_windows.size() == 0 ) return;
	size_t index = nv::random::get().urand( m_windows.size() );
	m_windows.erase( m_windows.begin() + index );
}

void application::recolor_window()
{
	if ( m_windows.size() == 0 ) return;
	size_t index = nv::random::get().urand( m_windows.size() );
	m_windows[ index ].change_color( nv::vec4( (float)rand() / float(RAND_MAX), (float)rand() / float(RAND_MAX), (float)rand() / float(RAND_MAX), 1.0 ) );
}

void application::simplify_window()
{
	if ( m_windows.size() == 0 ) return;
	size_t index = nv::random::get().urand( m_windows.size() );
	NV_LOG( nv::LOG_INFO, "Simplify " << index );
	m_windows[ index ].simplify_toggle();
}

application::~application()
{
	m_windows.clear();
	delete m_quad_cache;

	m_device->release( m_program );
	m_context->release( m_va );
	delete m_window;
	delete m_device;
}


int main(int, char* [])
{
	nv::random::get().randomize();
	nv::logger log(nv::LOG_TRACE);
	log.add_sink( new nv::log_file_sink("log.txt"), nv::LOG_TRACE );
	log.add_sink( new nv::log_console_sink(), nv::LOG_TRACE );
	
	NV_LOG( nv::LOG_NOTICE, "Logging started" );
	application app;
	if ( app.initialize() )
	{
		app.run();
	}
	NV_LOG( nv::LOG_NOTICE, "Logging stopped" );

	return 0;
}

