// Copyright (C) 2012-2014 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

#include "nv/formats/md3_loader.hh"

#include <glm/gtc/constants.hpp>
#include "nv/core/logging.hh"
#include <cstring>

using namespace nv;

// based on http://www.icculus.org/~phaethon/q3/formats/md3format.html#Surface

// assuming low-endian
#define MD3_MAX_FRAMES    1024
#define MD3_MAX_TAGS      16
#define MD3_MAX_SURFACES  32
#define MD3_MAX_SHADERS   256
#define MD3_MAX_VERTS     4096
#define MD3_MAX_TRIANGLES 8192
#define MD3_XYZ_SCALE     (1.0f/64.0f)

struct md3_vec3_t
{
	float xyz[3];
};

struct md3_header_t
{
	char   ident[4]; // IDP3
	sint32 version;  // 15
	uint8  name[64]; // path name
	sint32 flags;
	sint32 num_frames;     // Number of Frame objects, with a maximum of MD3_MAX_FRAMES. Current value of MD3_MAX_FRAMES is 1024.
	sint32 num_tags;       // Number of Tag objects, with a maximum of MD3_MAX_TAGS. Current value of MD3_MAX_TAGS is 16.
	sint32 num_surfaces;   // Number of Surface objects, with a maximum of MD3_MAX_SURFACES. Current value of MD3_MAX_SURFACES is 32.
	sint32 num_skins;      // Number of Skin objects. I should note that I have not seen an MD3 using this particular field for anything; this appears to be an artifact from the Quake 2 MD2 format. Surface objects have their own Shader field.
	sint32 ofs_frames;     // Relative offset from start of MD3 object where Frame objects start. The Frame objects are written sequentially, that is, when you read one Frame object, you do not need to seek() for the next object.
	sint32 ofs_tags;       // Relative offset from start of MD3 where Tag objects start. Similarly written sequentially.
	sint32 ofs_surfaces;   // Relative offset from start of MD3 where Surface objects start. Again, written sequentially.
	sint32 ofs_eof;        // Relative offset from start of MD3 to the end of the MD3 object. Note there is no offset for Skin objects.
};

struct md3_frame_t
{
	md3_vec3_t min_bounds;
	md3_vec3_t max_bounds;
	md3_vec3_t local_origin;
	float      radius;
	uint8      name[16];
};

struct md3_tag_t
{
	uint8      name[64];
	md3_vec3_t origin;
	md3_vec3_t axis[3];
};

struct md3_surface_header_t
{
	char   ident[4]; // IDP3
	uint8  name[64]; // path name
	sint32 flags;
	sint32 num_frames;
	sint32 num_shaders;
	sint32 num_verts;
	sint32 num_triangles;
	sint32 ofs_triangles;
	sint32 ofs_shaders;
	sint32 ofs_st;
	sint32 ofs_xyznormal;
	sint32 ofs_end;
};

struct md3_shader_t
{
	uint8  name[64]; 
	sint32 shader_index;
};

struct md3_triangle_t
{
	sint32 indexes[3];
};

struct md3_texcoord_t
{
	float  st[2];
};

struct md3_vertex_t
{
	sint16 x;
	sint16 y;
	sint16 z;
	uint16 normal;
};

struct md3_surface_t
{
	md3_surface_header_t header;
	md3_shader_t*        shaders;
	md3_triangle_t*      triangles;
	md3_texcoord_t*      st;
	md3_vertex_t*        vertices;
};

struct md3_t
{
	md3_header_t   header;
	md3_frame_t*   frames;
	md3_tag_t*     tags;
	md3_surface_t* surfaces;
	// extra information (not in md3 file)
	sint32         vertices_per_frame;
};

static bool check_md3_magic( char* magic )
{
	return magic[0] == 'I' && magic[1] == 'D' && magic[2] == 'P' && magic[3] == '3';
}

static void free_md3_surface( md3_surface_t * surface )
{
	delete[] surface->shaders;
	delete[] surface->triangles;
	delete[] surface->st;
	delete[] surface->vertices;
}

static void free_md3( md3_t * md3 )
{
	sint32 count = md3->header.num_surfaces;
	for ( sint32 i = 0; i < count; ++i )
	{
		free_md3_surface( &md3->surfaces[i] );
	}
	delete[] md3->frames;
	delete[] md3->tags;
	delete[] md3->surfaces;
}

static bool read_surface( md3_surface_t * surface, nv::stream& source )
{
	sint32 pos = static_cast< sint32 >( source.tell() );
	source.read( &surface->header, sizeof(md3_surface_header_t), 1 );

	if ( !check_md3_magic( surface->header.ident ) )          return false;
	if ( surface->header.num_frames    >  MD3_MAX_FRAMES )    return false;
	if ( surface->header.num_shaders   >  MD3_MAX_SHADERS )   return false;
	if ( surface->header.num_verts     >  MD3_MAX_VERTS )     return false;
	if ( surface->header.num_triangles >  MD3_MAX_TRIANGLES ) return false;

	surface->shaders   = new md3_shader_t  [ surface->header.num_shaders ];
	surface->vertices  = new md3_vertex_t  [ surface->header.num_verts * surface->header.num_frames ];
	surface->st        = new md3_texcoord_t[ surface->header.num_verts ];
	surface->triangles = new md3_triangle_t[ surface->header.num_triangles ];

	source.seek( pos + surface->header.ofs_shaders, origin::SET );
	source.read( surface->shaders, sizeof( md3_shader_t ), static_cast<size_t>( surface->header.num_shaders ) );

	source.seek( pos + surface->header.ofs_triangles, origin::SET );
	source.read( surface->triangles, sizeof( md3_triangle_t ), static_cast<size_t>( surface->header.num_triangles ) );

	source.seek( pos + surface->header.ofs_st, origin::SET );
	source.read( surface->st, sizeof( md3_texcoord_t ), static_cast<size_t>( surface->header.num_verts ) );

	source.seek( pos + surface->header.ofs_xyznormal, origin::SET );
	source.read( surface->vertices, sizeof( md3_vertex_t ), static_cast<size_t>( surface->header.num_verts * surface->header.num_frames ) );

	if ( source.tell() != static_cast<std::size_t>( pos + surface->header.ofs_end ) ) return false;

	return true;
}

static bool read_md3( md3_t * md3, nv::stream& source )
{
	md3->frames   = nullptr;
	md3->tags     = nullptr;
	md3->surfaces = nullptr;

	source.read( &md3->header, sizeof(md3_header_t), 1 );

	if ( !check_md3_magic( md3->header.ident ) )        return false;
	if ( md3->header.num_frames   >  MD3_MAX_FRAMES )   return false;
	if ( md3->header.num_tags     >  MD3_MAX_TAGS )     return false;
	if ( md3->header.num_surfaces >  MD3_MAX_SURFACES ) 
	{
		// to always have a safe free
		md3->header.num_surfaces = 0;
		return false;
	}

	md3->frames   = new md3_frame_t  [ md3->header.num_frames ];
	md3->tags     = new md3_tag_t    [ md3->header.num_tags * md3->header.num_frames ];
	md3->surfaces = new md3_surface_t[ md3->header.num_surfaces ];
	std::memset( md3->surfaces, 0, static_cast< size_t >( md3->header.num_surfaces ) * sizeof(md3_surface_t) );

	source.seek( md3->header.ofs_frames, origin::SET );
	source.read( md3->frames, sizeof( md3_frame_t ), static_cast<size_t>( md3->header.num_frames ) );

	source.seek( md3->header.ofs_tags, origin::SET );
	source.read( md3->tags, sizeof( md3_tag_t ), static_cast<size_t>( md3->header.num_tags * md3->header.num_frames ) );

	source.seek( md3->header.ofs_surfaces, origin::SET );
	md3->vertices_per_frame = 0;

	for ( sint32 i = 0; i < md3->header.num_surfaces; ++i )
	{
		if ( !read_surface( md3->surfaces + i, source ) ) return false;
		if ( md3->header.num_frames != md3->surfaces[i].header.num_frames ) return false;
		md3->vertices_per_frame += md3->surfaces[i].header.num_verts;
	}
	return true;
}

static inline vec3 md3_vec3( const md3_vec3_t& v )
{
//	return vec3( v.xyz[0], v.xyz[1], v.xyz[2] );
	return vec3( v.xyz[0], v.xyz[2], v.xyz[1] );
}

static inline vec2 md3_texcoord( const md3_texcoord_t& v )
{
	return vec2( v.st[0], v.st[1] );
}

static vec3 s_normal_cache[256*256];
static bool s_normal_ready = false;

md3_loader::md3_loader( bool merge_all )
	: m_md3( nullptr ), m_merge_all( merge_all )
{
	if ( !s_normal_ready )
	{
		float pi      = glm::pi<float>();
		float convert = (2 * pi) / 255.0f;
		int n = 0;
		for ( int lat = 0; lat < 256; ++lat )
		{
			float flat    = lat * convert;
			float sin_lat = glm::sin( flat );
			float cos_lat = glm::cos( flat );
			for ( int lng = 0; lng < 256; ++lng, ++n )
			{
				float flng    = lng * convert;
				float sin_lng = glm::sin( flng );
				float cos_lng = glm::cos( flng );
				s_normal_cache[n].x = cos_lat * sin_lng;
//				s_normal_cache[n].y = sin_lat * sin_lng;
//				s_normal_cache[n].z = cos_lng;
				s_normal_cache[n].z = sin_lat * sin_lng;
				s_normal_cache[n].y = cos_lng;
			}
		}

		s_normal_ready = true;
	}
}


nv::md3_loader::~md3_loader()
{
	if (m_md3 != nullptr)
	{
		free_md3( (md3_t*)(m_md3) );
		delete (md3_t*)m_md3;
	}
}

bool nv::md3_loader::load( stream& source )
{
	m_md3 = (void*)(new md3_t);
	if ( !read_md3( (md3_t*)m_md3, source ) )
	{
		return false;
	}
	return true;
}

nv::key_raw_channel* nv::md3_loader::load_tags( const std::string& tag )
{
	md3_t* md3 = (md3_t*)m_md3;
	key_raw_channel* result = key_raw_channel::create<md3_key>( md3->header.num_frames );
	// TODO: is this brain damaged in efficiency (loop nest order) or what?
	for ( sint32 f = 0; f < md3->header.num_frames; ++f )
	{
		for ( sint32 i = 0; i < md3->header.num_tags; ++i )
		{
			const md3_tag_t& rtag = md3->tags[i + md3->header.num_tags * f];
			std::string rname((char*)(rtag.name));
			if (rname == tag)
			{
				vec3 axisx  ( md3_vec3( rtag.axis[0] ) );
				vec3 axisz  ( md3_vec3( rtag.axis[1] ) );
				vec3 axisy  ( md3_vec3( rtag.axis[2] ) );
				vec3 origin ( md3_vec3( rtag.origin )  );
				((md3_key*)(result->data))[f].tform = transform( origin, quat( mat3( axisx, axisy, axisz ) ) );
			}
		}

	}
	return result;
}

struct vtx_md3_pn
{
	nv::vec3 position;
	nv::vec3 normal;
};

struct vtx_md3_t
{
	nv::vec2 texcoord;
};

mesh_data* nv::md3_loader::release_mesh_data( size_t index )
{
	mesh_data* data = new mesh_data;
	release_mesh_frame( data, -1, index );
	return data;
}

void nv::md3_loader::release_mesh_frame( mesh_data* data, sint32 frame, sint32 surface )
{
	md3_t* md3 = (md3_t*)m_md3;
	sint32 num_surfaces  = md3->header.num_surfaces;
	sint32 num_verts     = 0;
	sint32 current_frame = ( frame == -1 ? 0 : frame );
	sint32 frame_count   = ( frame == -1 ? md3->header.num_frames : 1 );
	sint32 current_surf  = ( surface == -1 ? 0 : surface );
	sint32 surf_count    = ( surface == -1 ? md3->header.num_surfaces : 1 );
	sint32 index_count   = 0;

	if ( surface >= 0 )
	{
		index_count = md3->surfaces[surface].header.num_triangles * 3;
		num_verts   = md3->surfaces[surface].header.num_verts;
	}
	else
		for ( sint32 i = 0; i < num_surfaces; ++i )
		{
			index_count += md3->surfaces[i].header.num_triangles * 3;
			num_verts   += md3->surfaces[i].header.num_verts;
		}

	mesh_raw_channel* mc_pn = mesh_raw_channel::create< vtx_md3_pn >( num_verts * frame_count );
	mesh_raw_channel* mc_t  = mesh_raw_channel::create< vtx_md3_t >( num_verts );
	mesh_raw_channel* ic = mesh_raw_channel::create_index< uint16 >( index_count );
	vtx_md3_pn* vtx_pn = (vtx_md3_pn*)mc_pn->data;
	vtx_md3_t*  vtx_t  = (vtx_md3_t*) mc_t->data;
	uint16*     icp    = (uint16*)ic->data;

	uint32 index  = 0;
	uint32 iindex = 0;
	sint32 index_base = 0;

	while ( surf_count > 0 )
	{
		const md3_surface_t& surface = md3->surfaces[ current_surf ];
		const uint32         vcount  = static_cast< uint32 >( surface.header.num_verts );
		const uint32         tcount  = static_cast< uint32 >( surface.header.num_triangles );

		for (uint32 j = 0; j < vcount; ++j )
		{
			vtx_t[index++].texcoord = md3_texcoord( surface.st[j] );
		}

		for (size_t j = 0; j < tcount; ++j )
		{
			const md3_triangle_t& t = surface.triangles[j];
			icp[iindex++] = static_cast< uint16 >( index_base + t.indexes[0] );
			icp[iindex++] = static_cast< uint16 >( index_base + t.indexes[1] );
			icp[iindex++] = static_cast< uint16 >( index_base + t.indexes[2] );
		}
		index_base += surface.header.num_verts;
		++current_surf;
		--surf_count;
	}

	index = 0;
	while ( frame_count > 0 )
	{
		current_surf  = ( surface == -1 ? 0 : surface );
		surf_count    = ( surface == -1 ? md3->header.num_surfaces : 1 );

		while ( surf_count > 0 )
		{
			md3_surface_t& surface = md3->surfaces[current_surf];
			sint32         vcount  = surface.header.num_verts;
			sint32         offset  = vcount * current_frame;
			sint32         limit   = vcount + offset;
			for (sint32 j = offset; j < limit; ++j )
			{
				md3_vertex_t& v = surface.vertices[j];
				vtx_pn[index].position = vec3( v.x * MD3_XYZ_SCALE, v.z * MD3_XYZ_SCALE, v.y * MD3_XYZ_SCALE );
				vtx_pn[index].normal   = s_normal_cache[ v.normal ];
				index++;
			}
			++current_surf;
			--surf_count;
		}
		++current_frame;
		--frame_count;
	}

	data->set_name( (char*)md3->header.name );
	data->add_channel( mc_pn );
	data->add_channel( mc_t );
	data->add_channel( ic );
}

mesh_nodes_data* nv::md3_loader::release_mesh_nodes_data( size_t )
{
	md3_t* md3 = (md3_t*)m_md3;
	uint32 node_count = md3->header.num_tags;
	if ( node_count == 0 ) return nullptr;;
	mesh_node_data* nodes = new mesh_node_data[ node_count ];
	for ( uint32 i = 0; i < node_count; ++i )
	{
		const md3_tag_t& rtag = md3->tags[i];
		std::string name( (char*)(rtag.name) );

		nodes[i].transform = mat4();
		nodes[i].name      = name;
		nodes[i].parent_id = -1;
		nodes[i].target_id = -1;
		nodes[i].data      = new key_data;
	
		key_raw_channel* keys = load_tags( name );
		nodes[i].data->add_channel( keys );
	}
	return new mesh_nodes_data( "tags", node_count, nodes );
}

mesh_data_pack* nv::md3_loader::release_mesh_data_pack()
{
	md3_t* md3 = (md3_t*)m_md3;
	uint32 count = 1;
	mesh_data* data = nullptr;
	if ( m_merge_all )
	{
		data = new mesh_data[1];
		release_mesh_frame( &data[0], -1, -1 );
		data[0].set_name( (char*)md3->header.name );
	}
	else
	{
		count = md3->header.num_surfaces;
		data = new mesh_data[ count ];
		for ( uint32 i = 0; i < count; ++i )
		{
			release_mesh_frame( &data[i], -1, i );
			data[i].set_name( (char*)md3->surfaces[i].header.name );
		}
	}
	return new mesh_data_pack( count, data, release_mesh_nodes_data() );
}

size_t md3_loader::get_max_frames() const
{
	return static_cast< size_t >( ((md3_t*)m_md3)->header.num_frames );
}
