// Copyright (C) 2012-2013 ChaosForge / Kornel Kisielewicz
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh
/**
 * @file context.hh
 * @author Kornel Kisielewicz epyon@chaosforge.org
 * @brief Context class
 */

#ifndef NV_CONTEXT_HH
#define NV_CONTEXT_HH

#include <nv/common.hh>
#include <nv/interface/device.hh>
#include <nv/interface/camera.hh>
#include <nv/interface/vertex_buffer.hh>
#include <nv/interface/clear_state.hh>
#include <nv/interface/render_state.hh>

namespace nv
{
	enum primitive
	{
		POINTS,
		LINES,
		LINE_LOOP,
		LINE_STRIP,
		TRIANGLES,
		TRIANGLE_STRIP,
		TRIANGLE_FAN
	};

	class mesh_interface
	{
	public:
		mesh_interface() {}
		virtual void update( uint32 ) {}
		virtual void update( program ) {}
		virtual nv::vertex_array get_vertex_array() const = 0;
		virtual size_t get_index_count() const = 0;
		virtual nv::primitive get_primitive_type() const { return nv::TRIANGLES; }
		virtual ~mesh_interface() {}
	};

	class context
	{
	public:
		context( device* a_device )	
		{ 
			m_device = a_device; 
		}
		virtual void bind( texture, texture_slot ) = 0;
		virtual void bind( buffer ) = 0;
		virtual void bind( program ) = 0;
		virtual void bind( vertex_array ) = 0;
		virtual void unbind( buffer ) = 0;
		virtual void unbind( program ) = 0;
		virtual void update( texture, void* ) = 0;
		virtual void update( buffer, const void*, size_t /*offset*/, size_t /*size*/ ) = 0;

		virtual void clear( const clear_state& cs ) = 0;
		// temporary
		virtual void draw( primitive prim, const render_state& rs, program p, vertex_array va, size_t count ) = 0;

		void draw( const render_state& rs, program p, const mesh_interface* mesh )
		{
			draw( mesh->get_primitive_type(), rs, p, mesh->get_vertex_array(), mesh->get_index_count() );
		}
		void draw( const render_state& rs, const scene_state& s, program p, const mesh_interface* mesh )
		{
			apply_engine_uniforms(p,s);
			draw( rs, p, mesh );
		}
		void draw( primitive prim, const render_state& rs, const scene_state& s, program p, vertex_array va, size_t count )
		{
			apply_engine_uniforms(p,s);
			draw( prim, rs, p, va, count );
		}
		virtual void apply_engine_uniforms( program p, const scene_state& s ) = 0;

		virtual void apply_render_state( const render_state& state ) = 0;
		virtual const ivec4& get_viewport() = 0;
		virtual void set_viewport( const ivec4& viewport ) = 0;
		virtual device* get_device() { return m_device; }
		virtual ~context() 
		{
		}
	protected:

		device*      m_device;
		clear_state  m_clear_state;
		render_state m_render_state;
		ivec4        m_viewport;
	};

} // namespace nv

#endif // NV_CONTEXT_HH
