// Copyright (C) 2012 Kornel Kisielewicz
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

#ifndef NV_TYPES_HH
#define NV_TYPES_HH

#include <nv/common.hh>
#include <utility>
#include <unordered_map>
#include <string>

namespace nv
{
	template <typename TYPE>
    const char* get_type_name()
    {
        static_assert< FALSE >( "Type not implemented!" );
    }

    template <> const char* get_type_name<sint8> () { return "sint8"; }
    template <> const char* get_type_name<sint16>() { return "sint16"; }
    template <> const char* get_type_name<sint32>() { return "sint32"; }
    template <> const char* get_type_name<sint64>() { return "sint64"; }

	template <> const char* get_type_name<uint8> () { return "uint8"; }
    template <> const char* get_type_name<uint16>() { return "uint16"; }
    template <> const char* get_type_name<uint32>() { return "uint32"; }
    template <> const char* get_type_name<uint64>() { return "uint64"; }

	template <> const char* get_type_name<f32> () { return "f32"; }
	template <> const char* get_type_name<f64> () { return "f64"; }

	template <> const char* get_type_name<bool> () { return "bool"; }

	template <> const char* get_type_name<std::string> () { return "string"; }

	template <typename TYPE>
    std::size_t get_type_id()
    {
        static std::size_t type_id = std::hash<std::string>()(std::string(get_type_name<TYPE>()));
        return type_id;
    };

	struct hash_string
    {
        std::size_t hash;
        const char* text;
		hash_string() {}
		hash_string( const char* a_text ) : 
			text( a_text ) 
		{
			hash = std::hash<std::string>()( std::string( a_text ) ); 
		}
		inline bool operator == (const hash_string& hs ) const
		{
			return hs.hash == hash;
		}
    };

}

namespace std {
	template<>
	struct hash<nv::hash_string> {
		size_t operator()(const nv::hash_string &hs) const 
		{
			return hs.hash;
		}
	};
}

namespace nv
{

    struct type
    {
		 // Function types for the constructor and destructor of registered types
	    typedef void (*constructor_func)(void*);
	    typedef void (*destructor_func)(void*);

		// Parent type database
        class type_database* type_db;

        // Scoped C++ name of the type
        hash_string name;
 
        // Pointers to the constructor and destructor functions
        constructor_func constructor;
        destructor_func  destructor;
 
        // Result of sizeof(type) operation
        size_t size;
    };
 
    class type_database
    {
    public:
		template< typename TYPE >
        type& create_type()
		{
			hash_string name( get_type_name<TYPE>() );
			type* i_type = nullptr;
			type_map::iterator it = m_types.find( name );
			if ( it != m_types.end() ) 
			{
				return *(it->second);
			}
			i_type          = new type;
			i_type->type_db = this;
			i_type->name    = name;
			i_type->size    = sizeof(TYPE);
			
			i_type->constructor = ConstructObject<TYPE>;
			i_type->destructor  = DestructObject<TYPE>;

			m_types[name] = i_type;
			return *i_type;
		}
        type* get_type( hash_string name )
		{
			type_map::iterator it = m_types.find( name );
			if ( it != m_types.end() ) 
			{
				return it->second;
			}
			return nullptr;
		}
    private:
        typedef std::unordered_map<hash_string, type*> type_map;
        type_map m_types;
	};

	template <typename TYPE> void ConstructObject(void* object)
    {
        // Use placement new to call the constructor
        new (object) TYPE;
    }
    template <typename TYPE> void DestructObject(void* object)
    {
        // Explicit call of the destructor
        ((TYPE*)object)->TYPE::~TYPE();
    }

}

#endif NV_TYPES_HH