// Copyright (C) 2014 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

#ifndef NV_ASSIMP_LOADER_HH
#define NV_ASSIMP_LOADER_HH

#include <nv/common.hh>
#include <nv/io/string_table.hh>
#include <nv/interface/mesh_loader.hh>
#include <nv/interface/mesh_data.hh>
#include <nv/interface/animated_mesh.hh>

namespace nv 
{
	struct assimp_key_p  { float time; vec3 position; };
	struct assimp_key_r  { float time; quat rotation; };
	struct assimp_key_s  { float time; vec3 scale; };
	struct assimp_key_tr { transform tform; };


	struct assimp_bone
	{
		string name;
		mat4   offset;

		assimp_bone(){}
		assimp_bone( const string& n, const mat4& m ) : name(n), offset(m) {}
	};

	struct assimp_model
	{
		std::vector< mesh_data* >  meshes;
		std::vector< assimp_bone > bones;
	};

	struct assimp_animated_node_data
	{
		std::string           name;
		sint32                parent_id;
		mat4                  transform;
		key_data*             data;

		assimp_animated_node_data() 
			: name(), parent_id( -1 ), data( nullptr )
		{
		}
		~assimp_animated_node_data() 
		{
			if ( data ) delete data;
		}
	};

	struct assimp_animation
	{
		float fps;
		float duration;
		bool  pretransformed;
		dynamic_array< assimp_animated_node_data > nodes;
	};

	class assimp_loader : public mesh_loader
	{
	public:
		assimp_loader( const string& a_ext, const mat3& a_rotate_transform, float a_scale, uint32 a_assimp_flags = 0 );
		explicit assimp_loader( const string& a_ext, uint32 a_assimp_flags = 0 );
		virtual bool load( stream& source );
		virtual mesh_data* release_mesh_data( size_t index = 0 );
		virtual size_t get_mesh_count() const { return m_mesh_count; }
		virtual ~assimp_loader();
		assimp_model* release_merged_model();
		assimp_animation* release_animation( size_t index, bool pre_transform );
		bool load_bones( size_t index, std::vector< assimp_bone >& bones );
		void scene_report() const;
	private:
		void initialize( const string& a_ext, const mat3& a_rotate_transform, float a_scale, uint32 a_assimp_flags );
		uint32 load_node( assimp_animation* data, const void* vnode, sint32 this_id, sint32 parent_id );
		uint32 count_nodes( const void* node ) const;
		void create_transformed_keys( assimp_animated_node_data* data, const void* vnode, const assimp_animated_node_data* parent );
		void create_direct_keys( assimp_animated_node_data* data, const void* vnode );

		string_table_creator m_strings;
		string m_ext;
		mat3   m_r33;
		mat3   m_ri33;
		float  m_scale;
		uint32 m_assimp_flags;
		size_t m_mesh_count;
		const void* m_scene;
	};

	struct assimp_plain_vtx 
	{
		vec3 position;
		vec3 normal;
		vec2 texcoord;
		vec4 tangent;

		assimp_plain_vtx() {}
		assimp_plain_vtx( const vec3& v, const vec2& t, const vec3& n, const vec4& g )
		{
			position = v;
			texcoord = t;
			normal   = n;
			tangent  = g;
		}
	};

	struct assimp_skinned_vtx 
	{
		vec3  position;
		vec3  normal;
		vec2  texcoord;
		vec4  tangent;
		ivec4 boneindex;
		vec4  boneweight;

		assimp_skinned_vtx() {}
		assimp_skinned_vtx( const vec3& v, const vec2& t, const vec3& n, const vec4& g )
		{
			position = v;
			texcoord = t;
			normal   = n;
			tangent  = g;
		}
	};

}

#endif // NV_ASSIMP_LOADER_HH
