// Copyright (C) 2014 ChaosForge / Kornel Kisielewicz
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

#ifndef NV_NMD_LOADER_HH
#define NV_NMD_LOADER_HH

#include <nv/common.hh>
#include <nv/interface/mesh_loader.hh>
#include <nv/interface/mesh_data.hh>
#include <nv/io/string_table.hh>

namespace nv 
{

	enum class nmd_type : uint16
	{
		MESH,
		STREAM,
		BONE_ARRAY,
		STRING_TABLE,
		ANIMATION,
		ANIMATION_NODE,
		ANIMATION_CHANNEL,
	};

	enum class nmd_key_type : uint16
	{
		NONE,
		TRANSFORM,
	};

	struct nmd_header
	{
		uint32 id;
		uint32 version;
		uint32 elements;
	};

	struct nmd_element_header
	{
		nmd_type type;
		uint16   children;
		uint32   size;
	};

	struct nmd_animation_header
	{
		float fps;
		float duration;
		bool  flat;
	};

	struct nmd_animation_node_header
	{
		uint16 name;
		sint16 parent_id;
		mat4   transform;
	};

	struct nmd_animation_channel_header
	{
		key_descriptor format;
		uint32         count;
	};

	struct nmd_stream_header
	{
		vertex_descriptor format;
		uint32 count;
	};

	struct nmd_bone
	{
		uint16 name;
		mat4   offset;
	};

	struct nmd_node
	{
		uint16 name;
		sint16 parent_id;
		mat4   transform;
		uint16 channel_count;
		key_raw_channel** channels;
	};

	struct nmd_animation
	{
		float     fps;
		float     duration;
		bool      flat;
		uint16    node_count;
		nmd_node* nodes;

		nmd_animation() : node_count(0), nodes( nullptr ) {}
		void clear_node( nmd_node* node )
		{
			if ( node->channel_count > 0 )
			{
				for ( uint16 c = 0; c < node->channel_count; ++c )
					delete node->channels[c];
				delete[] node->channels;
			}
			node->channels = nullptr;
			node->channel_count = 0;
		}
		~nmd_animation()
		{
			for ( uint16 n = 0; n < node_count; ++n )
				clear_node( &nodes[n] );
			delete[] nodes;
		}
	};

	struct nmd_bone_data
	{
		nmd_bone* bones;
		uint16    count;

		nmd_bone_data() : count(0), bones( nullptr ) {}
		~nmd_bone_data()
		{
			if ( bones ) delete[] bones;
		}
	};



	class nmd_loader : public mesh_loader
	{
	public:
		nmd_loader() : m_animation( nullptr ), m_bone_data( nullptr ), m_strings( nullptr ) {}
		virtual bool load( stream& source );
		virtual mesh_data* release_mesh_data( size_t index = 0 );
		virtual nmd_animation* release_animation();
		virtual nmd_bone_data* release_bone_data();
		virtual string_table* release_string_table();
		virtual size_t get_mesh_count() const { return m_meshes.size(); }
		virtual ~nmd_loader();
	private:
		void reset();
		bool load_mesh( stream& source, uint32 children );
		bool load_bones( stream& source, uint32 children ); 
		bool load_strings( stream& source ); 
		bool load_animation( stream& source, uint32 children );

		nmd_animation*            m_animation;
		nmd_bone_data*            m_bone_data;
		string_table*             m_strings;
		std::vector< mesh_data* > m_meshes;
	};

	// TODO: Temporary, find a better way and remove!
	struct nmd_temp_node_data
	{
		key_animation_data* keys;
		mat4                transform;
		sint32              bone_id;
		std::vector< nmd_temp_node_data* > children;
	};

	class nmd_temp_model;

	class nmd_temp_animation
	{
	public:
		nmd_temp_animation( nmd_loader* loader );
		void prepare( const nmd_temp_model* model );
		void animate( mat4* data, uint32 time );
		virtual uint32 get_frame_rate() const { return uint32(m_animation->fps); }
		virtual uint32 get_frame_count() const { return uint32(m_animation->duration) / uint32(m_animation->fps); }
		~nmd_temp_animation();
	private:
		void animate_rec( mat4* data, float time, uint32 node_id, const mat4& parent_mat );

		std::vector< key_animation_data* > m_data;
		std::vector< sint16 > m_bone_ids;
		std::vector< std::vector< uint32 > > m_children;
		const mat4*    m_offsets;
		string_table*  m_strings;
		nmd_animation* m_animation;
	};

	// TODO: Temporary, find a better way and remove!
	class nmd_temp_model
	{
		friend class nmd_temp_animation;
	public:
		nmd_temp_model( nmd_loader* loader );
		const mesh_data* get_data( uint32 index ) const { return m_mesh_data[index]; }
		uint32 get_bone_count() const { return m_bone_offsets.size(); }
		uint32 get_count() const { return m_mesh_data.size(); }
		~nmd_temp_model();
	private:
		std::unordered_map< std::string, nv::uint16 > m_bone_names;
		std::vector< mat4 >      m_bone_offsets;
		std::vector< mesh_data*> m_mesh_data;
	};

}

#endif // NV_NMD_LOADER_HH
