// Copyright (C) 2012-2014 ChaosForge / Kornel Kisielewicz
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh
/**
 * @file program.hh
 * @author Kornel Kisielewicz epyon@chaosforge.org
 * @brief Program class
 */

#ifndef NV_PROGRAM_HH
#define NV_PROGRAM_HH

#include <unordered_map>
#include <nv/interface/uniform.hh>
#include <nv/interface/vertex.hh>
#include <nv/logging.hh>
#include <nv/exception.hh>
#include <nv/common.hh>
#include <nv/string.hh>
#include <nv/math.hh>

namespace nv
{
	class camera;

	enum texture_slot
	{
		TEX_DIFFUSE  = 0,
		TEX_SPECULAR = 1,
		TEX_NORMAL   = 2,
		TEXTURE_0    = 0,
		TEXTURE_1    = 1,
		TEXTURE_2    = 2,
		TEXTURE_3    = 3,
		TEXTURE_4    = 4,
		TEXTURE_5    = 5,
		TEXTURE_6    = 6,
		TEXTURE_7    = 7,
	};


	class attribute
	{
	public:
		attribute( const string& name, int location, datatype type,	int length ) :
			m_name( name ),	m_location( location ),	m_type( type ), m_length( length ) {}
		const string& get_name() const { return m_name; }
		int get_location() const { return m_location; }
		datatype get_type() const { return m_type; }
		int get_length() const { return m_length; }

	protected:
		string   m_name;
		int      m_location;
		datatype m_type;
		int      m_length;
	};

	typedef std::unordered_map< string, attribute* >    attribute_map;

	class program
	{
	public:
		virtual void bind() = 0;
		virtual void unbind() = 0;
		virtual bool is_valid() const = 0;
		
		virtual ~program()
		{
			for ( auto& i : m_attribute_map )   delete i.second;
			for ( auto& i : m_uniform_map )     delete i.second;
		}

		const attribute_map& get_attributes() const { return m_attribute_map; }
		const uniform_map& get_uniforms() const { return m_uniform_map; }

		attribute* try_get_attribute( const string& name ) const
		{
			attribute_map::const_iterator i = m_attribute_map.find( name );
			if ( i != m_attribute_map.end() )
			{
				return i->second;
			}
			return nullptr;
		}

		int try_get_attribute_location( const string& name ) const
		{
			attribute_map::const_iterator i = m_attribute_map.find( name );
			if ( i != m_attribute_map.end() )
			{
				return i->second->get_location();
			}
			return -1;
		}

		attribute* get_attribute( const string& name ) const
		{
			attribute_map::const_iterator i = m_attribute_map.find( name );
			if ( i != m_attribute_map.end() )
			{
				return i->second;
			}
			NV_LOG( LOG_ERROR, "Attribute '" << name << "' not found in program!" );
			NV_THROW( runtime_error, ( "Attribute '"+name+"' not found!" ) );
		}

		uniform_base* try_get_uniform( const string& name ) const
		{
			uniform_map::const_iterator i = m_uniform_map.find( name );
			if ( i != m_uniform_map.end() )
			{
				return i->second;
			}
			return nullptr;
		}

		uniform_base* get_uniform( const string& name ) const
		{
			uniform_map::const_iterator i = m_uniform_map.find( name );
			if ( i != m_uniform_map.end() )
			{
				return i->second;
			}
			NV_LOG( LOG_ERROR, "Uniform '" << name << "' not found in program!" );
			NV_THROW( runtime_error, ( "Uniform '"+name+"' not found!" ) );
		}

		template < typename T >
		void set_uniform_array( const string& name, const T* value, uint32 count )
		{
			uniform_base* base = get_uniform( name );
			// restore typechecking, but remember to accept int for float!
			// if ( /* base->get_type() != type_to_enum<T>::type */ )
			// {
			//		NV_LOG( LOG_ERROR, "Uniform '" << name << "' not found in program!" );
			//		return;
			// }
			// TODO: nicer check
			NV_ASSERT( count <= base->get_length(), "LENGTH CHECK FAIL" );
			((uniform<T>*)( base ))->set_value( value, count );
		}

		template < typename T >
		void set_uniform_array( const string& name, const std::vector<T>& value )
		{
			uniform_base* base = get_uniform( name );
			// restore typechecking, but remember to accept int for float!
			// if ( /* base->get_type() != type_to_enum<T>::type */ )
			// {
			//		NV_LOG( LOG_ERROR, "Uniform '" << name << "' not found in program!" );
			//		return;
			// }
			// TODO: nicer check
			NV_ASSERT( (int)value.size() <= base->get_length(), "LENGTH CHECK FAIL" );
			((uniform<T>*)( base ))->set_value( value.data(), value.size() );
		}

		template < typename T >
		void set_opt_uniform_array( const string& name, const T* value, uint32 count )
		{
			uniform_base* base = try_get_uniform( name );
			if (!base) return;
			// restore typechecking, but remember to accept int for float!
			// if ( /* base->get_type() != type_to_enum<T>::type */ )
			// {
			//		NV_LOG( LOG_ERROR, "Uniform '" << name << "' not found in program!" );
			//		return;
			// }
			// TODO: nicer check
			NV_ASSERT( (int)count <= base->get_length(), "LENGTH CHECK FAIL" );
			((uniform<T>*)( base ))->set_value( value, count );
		}

		template < typename T >
		void set_opt_uniform_array( const string& name, const std::vector<T>& value )
		{
			uniform_base* base = try_get_uniform( name );
			if (!base) return;
			// restore typechecking, but remember to accept int for float!
			// if ( /* base->get_type() != type_to_enum<T>::type */ )
			// {
			//		NV_LOG( LOG_ERROR, "Uniform '" << name << "' not found in program!" );
			//		return;
			// }
			// TODO: nicer check
			NV_ASSERT( (int)value.size() <= base->get_length(), "LENGTH CHECK FAIL" );
			((uniform<T>*)( base ))->set_value( value.data(), value.size() );
		}


		template < typename T >
		void set_uniform( const string& name, const T& value )
		{
			uniform_base* base = try_get_uniform( name );
			// restore typechecking, but remember to accept int for float!
			// if ( /* base->get_type() != type_to_enum<T>::type */ )
			// {
			//		NV_LOG( LOG_ERROR, "Uniform '" << name << "' not found in program!" );
			//		return;
			// }
			((uniform<T>*)( base ))->set_value( value );
		}

		template < typename T >
		void set_opt_uniform( const string& name, const T& value )
		{
			uniform_base* base = try_get_uniform( name );
			if ( base != nullptr )
			{
				// restore typechecking, but remember to accept int for float!
				// if ( /* base->get_type() != type_to_enum<T>::type */ )
				// {
				//		NV_LOG( LOG_ERROR, "Uniform '" << name << "' not found in program!" );
				//		return;
				// }
				((uniform<T>*)( base ))->set_value( value );
			}
		}

		void apply_engine_uniforms( const context* ctx, const scene_state* s )
		{
			for ( auto u : m_engine_uniforms )
			{
				u->set( ctx, s );
			}
		}

		void apply_link_engine_uniforms()
		{
			engine_link_uniform_factory_map& factory_map = get_link_uniform_factory();
			for ( auto& i : m_uniform_map )
			{
				auto j = factory_map.find( i.first );
				if ( j != factory_map.end() )
				{
					j->second->set( i.second );
				}				
			}
		}

		void bind_engine_uniforms()
		{
			engine_uniform_factory_map& factory_map = get_uniform_factory();
			for ( auto& i : m_uniform_map )
			{
				auto j = factory_map.find( i.first );
				if ( j != factory_map.end() )
				{
					m_engine_uniforms.push_back( j->second->create( i.second ) );
				}				
			}
		}

		// This is done this way to avoid compilation unit creation
		static engine_uniform_factory_map& get_uniform_factory()
		{
			static engine_uniform_factory_map s_engine_uniform_factory_map;
			return s_engine_uniform_factory_map;
		}

		// This is done this way to avoid compilation unit creation
		static engine_link_uniform_factory_map& get_link_uniform_factory()
		{
			static engine_link_uniform_factory_map s_engine_link_uniform_factory_map;
			return s_engine_link_uniform_factory_map;
		}

		static void initialize_engine_uniforms()
		{
			engine_uniform_factory_map& factory_map = get_uniform_factory();
			factory_map[ "nv_m_view" ]       = new engine_uniform_factory< engine_uniform_m_view >();
			factory_map[ "nv_m_view_inv" ]   = new engine_uniform_factory< engine_uniform_m_view_inv >();
			factory_map[ "nv_m_model" ]      = new engine_uniform_factory< engine_uniform_m_model >();
			factory_map[ "nv_m_model_inv" ]  = new engine_uniform_factory< engine_uniform_m_model_inv >();
			factory_map[ "nv_m_modelview" ]  = new engine_uniform_factory< engine_uniform_m_modelview >();
			factory_map[ "nv_m_projection" ] = new engine_uniform_factory< engine_uniform_m_projection >();
			factory_map[ "nv_m_normal" ]     = new engine_uniform_factory< engine_uniform_m_normal >();
			factory_map[ "nv_m_mvp" ]        = new engine_uniform_factory< engine_uniform_m_mvp >();
			factory_map[ "nv_v_camera_position" ]  = new engine_uniform_factory< engine_uniform_v_camera_position >();
			factory_map[ "nv_v_camera_direction" ] = new engine_uniform_factory< engine_uniform_v_camera_direction >();

			engine_link_uniform_factory_map& factory_link_map = get_link_uniform_factory();
			factory_link_map[ "nv_texture_0" ] = new engine_link_uniform_int<0>();
			factory_link_map[ "nv_texture_1" ] = new engine_link_uniform_int<1>();
			factory_link_map[ "nv_texture_2" ] = new engine_link_uniform_int<2>();
			factory_link_map[ "nv_texture_3" ] = new engine_link_uniform_int<3>();
			factory_link_map[ "nv_texture_4" ] = new engine_link_uniform_int<4>();
			factory_link_map[ "nv_texture_5" ] = new engine_link_uniform_int<5>();
			factory_link_map[ "nv_texture_6" ] = new engine_link_uniform_int<6>();
			factory_link_map[ "nv_texture_7" ] = new engine_link_uniform_int<7>();
			factory_link_map[ "nv_t_diffuse" ] = new engine_link_uniform_int<0>();
			factory_link_map[ "nv_t_specular"] = new engine_link_uniform_int<1>();
			factory_link_map[ "nv_t_normal"  ] = new engine_link_uniform_int<2>();
		}

		static void destroy_engine_uniforms()
		{
			for ( auto& i : get_uniform_factory() ) delete i.second;
			for ( auto& i : get_link_uniform_factory() ) delete i.second;
			get_uniform_factory().clear();
			get_link_uniform_factory().clear();
		}

	protected:

		uniform_base* create_uniform( datatype utype, const string& name, int location, int length )
		{
			switch( utype )
			{
			case FLOAT          : return new uniform< enum_to_type< FLOAT          >::type >( name, location, length );
			case INT            : return new uniform< enum_to_type< INT            >::type >( name, location, length );
			case FLOAT_VECTOR_2 : return new uniform< enum_to_type< FLOAT_VECTOR_2 >::type >( name, location, length );
			case FLOAT_VECTOR_3 : return new uniform< enum_to_type< FLOAT_VECTOR_3 >::type >( name, location, length );
			case FLOAT_VECTOR_4 : return new uniform< enum_to_type< FLOAT_VECTOR_4 >::type >( name, location, length );
			case INT_VECTOR_2   : return new uniform< enum_to_type< INT_VECTOR_2   >::type >( name, location, length );
			case INT_VECTOR_3   : return new uniform< enum_to_type< INT_VECTOR_3   >::type >( name, location, length );
			case INT_VECTOR_4   : return new uniform< enum_to_type< INT_VECTOR_4   >::type >( name, location, length );
			case FLOAT_MATRIX_2 : return new uniform< enum_to_type< FLOAT_MATRIX_2 >::type >( name, location, length );
			case FLOAT_MATRIX_3 : return new uniform< enum_to_type< FLOAT_MATRIX_3 >::type >( name, location, length );
			case FLOAT_MATRIX_4 : return new uniform< enum_to_type< FLOAT_MATRIX_4 >::type >( name, location, length );
			default     : return nullptr;
			}
		}

		attribute_map                     m_attribute_map;
		uniform_map	                      m_uniform_map;
		engine_uniform_list               m_engine_uniforms;
	};

} // namespace nv

#endif // NV_PROGRAM_HH
