// Copyright (C) 2012 Kornel Kisielewicz
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

#ifndef NV_TYPES_HH
#define NV_TYPES_HH

#include <nv/common.hh>
#include <nv/math.hh>
#include <nv/object.hh>
#include <nv/type_traits.hh>
#include <typeinfo>
#include <typeindex>
#include <utility>
#include <unordered_map>
#include <vector>
#include <string>
#include <cstddef>

namespace nv
{

	struct type_entry;
	class type_database;

	enum type_flag
	{
		TF_POINTER      = 0x01, //!< field is a pointer
		TF_NOSERIALIZE  = 0x02, //!< ignore during serialization
		TF_INVISIBLE    = 0x04, //!< field invisible to API
		TF_READONLY     = 0x08, //!< read only field
		TF_SIMPLETYPE   = 0x10, //!< raw binary I/O possible
		TF_OWNED        = 0x20,
		TF_CONTAINER    = 0x40, //!< is a container
	};

	struct type_field
	{
		std::string      name;          //!< name of the field
		std::string      type_name;     //!< name of the type of the field
		const std::type_info* type_inf; //!< typeinfo for later retrieval of type
		type_entry*      type;          //!< pointer to field type
		unsigned int     flags;         //!< flags 
		size_t           offset;

		template< typename TOBJECT, typename TFIELD >
		type_field( const char* aname, TFIELD TOBJECT::*field, typename std::enable_if< is_container<TFIELD>::value, void* >::type = nullptr )
		: name(aname)
			, type_name()
			, type_inf( &typeid( typename std::remove_pointer<typename TFIELD::value_type>::type ) )
			, type( nullptr )
			, flags( 0 )
			, offset( offset_of( field ) ) 
		{
			NV_LOG( LOG_INFO, aname << "-" << offset);
			flags = TF_CONTAINER |
				( std::is_pointer<typename TFIELD::value_type>::value ? TF_POINTER : 0 ) |
				( std::is_pod<typename TFIELD::value_type>::value ? TF_SIMPLETYPE : 0 );
		}

		template< typename TOBJECT, typename TFIELD >
		type_field( const char* aname, TFIELD TOBJECT::*field, typename std::enable_if< !is_container<TFIELD>::value, void* >::type = nullptr )
		: name(aname)
			, type_name()
			, type_inf( &typeid( typename std::remove_pointer<TFIELD>::type ) )
			, type( nullptr )
			, flags( 0 )
			, offset( offset_of( field ) )
		{
			NV_LOG( LOG_INFO, aname << "-" << offset);
			flags = 
				( std::is_pointer<TFIELD>::value ? TF_POINTER : 0 ) |
				( std::is_pod<TFIELD>::value ? TF_SIMPLETYPE : 0 );
		}

		type_field& flag( unsigned int f )
		{
			flags |= f;
			return *this;
		}
	};

	struct type_enum
	{
		std::string name;
		int         value;
		type_enum( const char* aname, int avalue ) : name(aname), value(avalue) {}
	};

	struct type_entry
	{
		// Function types for the constructor and destructor of registered types
		typedef void (*constructor_func)(void*);
		typedef void (*destructor_func)(void*);

		// Parent type database
		type_database* type_db;

		// Scoped C++ name of the type
		std::string name;

		// Pointers to the constructor and destructor functions
		constructor_func constructor;
		destructor_func  destructor;

		// Result of sizeof(type) operation
		size_t size;

		// Base type
		type_entry* base_type;

		// Field list
		std::vector<type_field> field_list;

		// Enum list
		std::vector<type_enum> enum_list;

		template <typename TYPE>
		type_entry& base();

		template <int SIZE>
		type_entry& fields( type_field (&init_fields)[SIZE] );

		template <int SIZE>
		type_entry& enums( type_enum (&init_enums)[SIZE] )
		{
			for (int i = 0; i < SIZE; i++)
			{
				enum_list.push_back( init_enums[i] );
			}
			return *this;
		}
	};

	class type_database
	{
	public:
		template< typename TYPE >
		type_entry& create_type( const char* name )
		{
			type_entry* i_type = nullptr;
			type_name_map::iterator it = m_name_types.find( name );
			if ( it != m_name_types.end() ) 
			{
				return *(it->second);
			}
			i_type          = new type_entry;
			i_type->type_db = this;
			i_type->name    = name;
			i_type->size    = sizeof(TYPE);

			i_type->constructor = construct_object<TYPE>;
			i_type->destructor  = destroy_object<TYPE>;

			m_name_types[name]        = i_type;
			m_idx_types[typeid(TYPE)] = i_type;
			return *i_type;
		}

		type_entry* get_type( const std::string name )
		{
			type_name_map::iterator it = m_name_types.find( name );
			if ( it != m_name_types.end() ) 
			{
				return it->second;
			}
			return nullptr;
		}

		type_entry* get_type( const std::type_info& t )
		{
			type_info_map::iterator it = m_idx_types.find( std::type_index(t) );
			if ( it != m_idx_types.end() ) 
			{
				return it->second;
			}
			return nullptr;
		}
	private:
		struct compare_type_info {
			bool operator ()(const std::type_info* a, const std::type_info* b) const {
				return a->before(*b);
			}
		};

		typedef std::unordered_map<std::string, type_entry*>     type_name_map;
		typedef std::unordered_map<std::type_index, type_entry*> type_info_map;
		type_name_map m_name_types;
		type_info_map m_idx_types;
	};
    
    template < typename TYPE >
    type_entry& type_entry::base()
    {
        base_type = type_db->get_type( typeid(TYPE) );
		return *this;
    }
    
    template <int SIZE>
    type_entry& type_entry::fields( type_field (&init_fields)[SIZE] )
    {
        for (int i = 0; i < SIZE; i++)
        {
            type_field f = init_fields[i];
            f.type = type_db->get_type(*(f.type_inf));
            f.type_name = f.type->name;
            field_list.push_back(f);
        }
        return *this;
    }

}

#endif

