// Copyright (C) 2012-2013 ChaosForge / Kornel Kisielewicz
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

#ifndef NV_MESH_DATA_HH
#define NV_MESH_DATA_HH

#include <nv/common.hh>
#include <vector>
#include <nv/math.hh>
#include <nv/interface/vertex.hh>

namespace nv
{

	// TODO: friend mesh_data_creator class?
	// TODO: private const etc


	struct mesh_raw_channel
	{
		vertex_descriptor desc;
		uint8*            data;
		uint32            size;
		uint32            count;

		mesh_raw_channel() : data( nullptr ), size( 0 ) {}
		~mesh_raw_channel() 
		{
			if ( data != nullptr ) delete[] data;
		}

		template < typename VTX >
		static mesh_raw_channel* create( uint32 count = 0 )
		{
			mesh_raw_channel* result = new mesh_raw_channel();
			result->desc.initialize<VTX>();
			result->count = count;
			result->size  = count * sizeof( VTX );
			result->data  = (count > 0 ? ( new uint8[ result->size ] ) : nullptr );
			return result;
		}
		static mesh_raw_channel* create( const vertex_descriptor& vtxdesc, uint32 count = 0 )
		{
			mesh_raw_channel* result = new mesh_raw_channel();
			result->desc  = vtxdesc;
			result->count = count;
			result->size  = count * sizeof( vtxdesc.size );
			result->data  = (count > 0 ? ( new uint8[ result->size ] ) : nullptr );
			return result;
		}
	};

	struct mesh_raw_index_channel
	{
		datatype          etype;
		uint8*            data;
		uint32            size;
		uint32            count;

		mesh_raw_index_channel() : etype( NONE ), data( nullptr ), size( 0 ) {}
		~mesh_raw_index_channel() 
		{
			if ( data != nullptr ) delete[] data;
		}

		template < typename ITYPE >
		static mesh_raw_index_channel* create( uint32 count = 0 )
		{
			mesh_raw_index_channel* result = new mesh_raw_index_channel();
			result->etype = type_to_enum< ITYPE >::type;
			result->count = count;
			result->size  = count * sizeof( ITYPE );
			result->data  = (count > 0 ? ( new uint8[ result->size ] ) : nullptr );
			return result;
		}

		static mesh_raw_index_channel* create( datatype etype, uint32 count = 0 )
		{
			mesh_raw_index_channel* result = new mesh_raw_index_channel();
			result->etype = etype;
			result->count = count;
			result->size  = count * get_datatype_info( etype ).size;
			result->data  = (count > 0 ? ( new uint8[ result->size ] ) : nullptr );
			return result;
		}
	};

	class mesh_data
	{
	public:
		mesh_data() : m_index_channel( nullptr ) {}
		void add_channel( mesh_raw_channel* channel ) { m_channels.push_back( channel ); }
		void set_index_channel( mesh_raw_index_channel* channel )
		{
			if ( m_index_channel ) delete m_index_channel;
			m_index_channel = channel;
		}

		const std::vector< mesh_raw_channel* >& get_channel_data() const { return m_channels; }
		const mesh_raw_index_channel*           get_index_channel() const { return m_index_channel; }

		size_t get_count() const 
		{
			if ( m_index_channel ) return m_index_channel->count;
			if ( m_channels.size() > 0 ) return m_channels[0]->count;
			return 0;
		}

		size_t get_count( size_t channel ) const 
		{
			if ( m_channels.size() > channel ) return m_channels[channel]->count;
			return 0;
		}

		virtual ~mesh_data()
		{
			for ( auto channel : m_channels ) delete channel;
			if ( m_index_channel ) delete m_index_channel;
		}
	private:
		std::vector< mesh_raw_channel* > m_channels;
		mesh_raw_index_channel*          m_index_channel;
	};


}

#endif // NV_MESH_DATA_HH
