// Copyright (C) 2014 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh
/**
 * @file range.hh
 * @author Kornel Kisielewicz epyon@chaosforge.org
 * @brief range iterators
 */

#ifndef NV_RANGE_HH
#define NV_RANGE_HH

#include <nv/common.hh>
#include <nv/math.hh>
#include <nv/type_traits.hh>
#include <iterator>

namespace nv
{

	namespace detail
	{

		template < typename T >
		class forward_iterator_base : public std::iterator< std::input_iterator_tag, T >
		{
		public:
			forward_iterator_base( T value ) : m_value( value ) {}
			T operator* () const { return m_value; }
			T const* operator-> () const { return &m_value; }
			bool operator== ( const forward_iterator_base& rhs ) const
			{
				return m_value == rhs.m_value;
			}
			bool operator!= ( const forward_iterator_base& rhs ) const
			{
				return !( *this == rhs );
			}
		protected:
			T m_value;
		};

		template < typename T >
		class range_iterator_base : public forward_iterator_base< T >
		{
		public:
			range_iterator_base( T value ) : forward_iterator_base( value ) {}
			range_iterator_base& operator++ () 
			{
				++m_value; 
				return *this; 
			}
			range_iterator_base operator++ (int) 
			{
				auto result = *this; 
				++*this; 
				return result;
			}
		};

		template < typename T >
		class range2d_iterator_base 
			: public forward_iterator_base< glm::detail::tvec2<T> >
		{
		public:
			range2d_iterator_base( glm::detail::tvec2<T> value, T min, T max ) 
				: forward_iterator_base( value ), m_min(min), m_max(max) {}
			range2d_iterator_base& operator++ () 
			{
				++m_value.x;
				if ( m_value.x > m_max ) 
				{
					++m_value.y;
					m_value.x = m_min;
				}
				return *this; 
			}
			range2d_iterator_base operator++ (int) 
			{
				auto result = *this; 
				++*this; 
				return result;
			}
		protected:
			T m_min;
			T m_max;
		};

	} // namespace detail

	template < typename T >
	class range_iterator_provider
	{
	public:
		class iterator : public detail::range_iterator_base<T>
		{
		public:
			iterator( T value ) : detail::range_iterator_base<T>(value) {}
		};

		range_iterator_provider( T begin, T end ) : m_begin( begin ), m_end( end ) {}
		iterator begin() { return m_begin; }
		iterator end() { return m_end; }

	protected:
		iterator m_begin;
		iterator m_end;
	};

	template < typename T >
	class range2d_iterator_provider
	{
	public:
		typedef T value_type;
		typedef typename T::value_type element_type;

		class iterator : public detail::range2d_iterator_base<element_type>
		{
		public:
			iterator( value_type begin, element_type min, element_type max ) 
				: detail::range2d_iterator_base<element_type>(begin, min, max) {}
		};

		range2d_iterator_provider( value_type begin, value_type end ) : m_begin( begin, begin.x, end.x ), m_end( T(begin.x, ++end.y ), begin.x, end.x ) {}
		iterator begin() { return m_begin; }
		iterator end() { return m_end; }

	protected:
		iterator m_begin;
		iterator m_end;
	};

	template < typename T >
	range_iterator_provider<T> range( T begin, T end )
	{
		return range_iterator_provider<T>( begin, end+1 );
	}

	template < typename T >
	range2d_iterator_provider<T> range2d( T begin, T end )
	{
		return range2d_iterator_provider<T>( begin, end );
	}

	template < typename T >
	range_iterator_provider<T> range( T end )
	{
		return range_iterator_provider<T>( 0, end );
	}

	template < typename T >
	range2d_iterator_provider<T> range2d( T end )
	{
		return range2d_iterator_provider<T>( T(0,0), T( --end.x, --end.y ) );
	}

	template < typename C >
	auto index( const C& c ) -> range_iterator_provider<decltype( c.size() )>
	{
		return range_iterator_provider( 0, c.size() );
	}

}

#endif // NV_RANGE_HH
