// Copyright (C) 2014 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

/**
 * @file handle.hh
 * @author Kornel Kisielewicz
 */

#ifndef NV_HANDLE_HH
#define NV_HANDLE_HH

#include <nv/common.hh>
#include <nv/array.hh>

namespace nv
{

	template < 
		typename T = uint32, 
		unsigned IBITS = 16,
		unsigned CBITS = 16,
		typename TAG = void 
	>
	class handle
	{
	public:
		typedef T value_type;
		static const int INDEX_BITS   = IBITS;
		static const int COUNTER_BITS = IBITS;
		static const T MAX_INDEX   = (1 << IBITS) - 1;
		static const T MAX_COUNTER = (1 << CBITS) - 1;

		handle() : m_index(0), m_counter(0) {}

		inline bool operator==(const handle& rhs){return m_index == rhs.m_index && m_counter == rhs.m_counter; }
		inline bool operator!=(const handle& rhs){return !(*this == rhs);}

		bool is_nil() const { return m_index == 0 && m_counter == 0; }
		bool is_valid() const { return !is_nil(); }

	protected:
		T m_index   : IBITS;
		T m_counter : CBITS;

		handle( T a_index, T a_counter ) : m_index( a_index ), m_counter( a_counter ) {}
		template < typename H, typename I >
		friend class index_store;
	};

	template < typename HANDLE, typename TINDEX = sint32 >
	class index_store
	{
	public:
		typedef HANDLE handle;
		typedef TINDEX index_type;
		typedef typename HANDLE::value_type value_type;

		index_store() : m_first_free(-1), m_last_free(-1) {}

		handle create_handle( index_type index )
		{
			value_type i       = get_free_entry();
			m_entries[i].index = index;
			m_entries[i].counter++;
			return handle( i, m_entries[i].counter );
		}

		void free_handle( handle h )
		{
			m_entries[ h.m_index ].index     = -1;
			m_entries[ h.m_index ].next_free = -1;
			if ( m_last_free == -1 )
			{
				m_first_free = m_last_free = h.m_index;
				return;
			}
			m_entries[ m_last_free ].next_free = h.m_index;
			m_last_free = h.m_index;
		}

		void swap_indices( handle h1, handle h2 )
		{
			std::swap( m_entries[ h1.m_index ].index, m_entries[ h2.m_index ].index );
		}

		sint32 get_index( handle h ) const
		{
			return m_entries[ h.m_index ].counter == h.m_counter ? m_entries[ h.m_index ].index : -1;
		}
	private:
		struct index_entry
		{
			index_type index;
			value_type counter;
			index_type next_free;

			index_entry() : index(0), counter(0), next_free(-1) {}
		};

		value_type get_free_entry()
		{
			if ( m_first_free != -1 )
			{
				value_type result = (value_type)m_first_free;
				m_first_free = m_entries[result].next_free;
				m_entries[result].next_free = -1;
				if ( m_first_free == -1 ) m_last_free = -1;
				return result;
			}
			m_entries.emplace_back();
			return value_type( m_entries.size() - 1 );
		}

		index_type m_first_free;
		index_type m_last_free;
		std::vector< index_entry > m_entries;
	};


}

#endif // NV_HANDLE_HH
