// Copyright (C) 2012-2013 ChaosForge / Kornel Kisielewicz
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

/**
 * @file uid.hh
 * @author Kornel Kisielewicz
 * @brief Implements a unique identifier class manager
 */

#ifndef NV_UID_HH
#define NV_UID_HH

#include <nv/common.hh>
#include <unordered_map>

namespace nv
{
	class object;

	class uid_store
	{
	public:
		
		/**
		 * Creates a new instance of the unique indentifier store.
		 */
		uid_store();

		/**
		 * Gets the object with the specified ID.
		 *
		 * @param auid The ID to fetch.
		 * @returns The stored object for that ID.
		 */
		object* get( uid auid ) const;

		/**
		 * Removes the object with the specified ID.
		 *
		 * @param auid The ID to remove.
		 * @returns True if the removal was successful, false if the ID didn't exist.
		 */
		bool remove( uid auid );

		/**
		 * Adds an object to the store assigned to the indicated ID.
		 *
		 * @param o The object to add to the store.
		 * @param auid The ID to assign to the object.
		 */
		void insert( object* o, uid auid );

		/**
		 * Adds an object to the store and assigns it a new ID.
		 *
		 * @param o The object to add to the store.
		 * @returns The ID the object was store under.
		 */
		uid insert( object* o );

		/**
		 * Gets the next available ID.
		 *
		 * @returns The next ID in the store.
		 */
		uid request_uid();

		/**
		 * Destroys the unique identifier store.
		 */
		~uid_store();

		/**
		 * Retrieves an object and casts it to the specified type.
		 *
		 * @tparam T The type to cast to.
		 * @param auid The ID the object is stored under.
		 * @returns An object of the indicated type that was stored at the indicated ID.
		 */
		template< typename T >
		T* get_as( uid auid ) const 
		{
			return dynamic_cast< T* >( get(auid) );
		}
	private:
		typedef std::unordered_map< uid, object* > map;
		map m_map; ///< The hash map everything is stored in.
		uid m_current; ///< The last UID assigned.
	};
}

#endif // NV_UID_HH
