// Copyright (C) 2012 Kornel Kisielewicz
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

#ifndef NV_OBJECT_HH
#define NV_OBJECT_HH

#include <nv/common.hh>
#include <nv/string.hh>
#include <list>

namespace nv
{

	/**
	 * Implements a object tree-like structure.
	 */
	class object
	{
	public:
		friend class root;

		/**
		 * Register the object type in the type database.
		 */
		//static void register_type( type_database* db );

		/// List type
		typedef std::list<object*> list;

		/**
		 * Object constructor
		 */
		explicit object( const string& aid );

		/**
		 * Adds a child to this object. 
		 *
		 * The child is added to the end of the list. The child is
		 * first detached from it's parent if present.
		 *
		 * @param child child to be added.
		 */
		virtual void add_child( object* child );


		/**
		 * Removes a child from this object.
		 *
		 * Does nothing if child not present, or is not child of
		 * this object.
		 *
		 * @param child child to be removed.
		 */
		virtual void remove_child( object* child );

		/**
		 * Detaches object from parent.
		 */
		void detach(); 

		/**
		 * Moves the node to a new parent.
		 *
		 * Simply calls detach, and then add_child( new_parent ).
		 *
		 * @param new_parent the parent where we move
		 */
		virtual void change_parent( object* new_parent );

		/**
		 * Templated function to return parent.
		 *
		 * If parent doesn't exist, or is of other type nullptr will be returned.
		 *
		 * @tparam T requested parent type.
		 * @returns parent as passed type, or nullptr if not present/invalid
		 */
		template< typename T >
		//typename T* get_parent_as() const
		T* get_parent_as() const
		{
			if ( !m_parent )
			{
				return nullptr;
			}
			return dynamic_cast<T>( m_parent );
		}

		/**
		 * Searches for child by pointer.
		 *
		 * @param child child being searched
		 * @param recursive whether to search recursively
		 * @returns nullptr if not found, child otherwise
		 */
		object* find( object* child, bool recursive = false );

		/**
		 * Searches for child by uid.
		 *
		 * @param child uid being searched
		 * @param recursive whether to search recursively
		 * @returns nullptr if not found, child otherwise
		 */
		object* find( uid child, bool recursive = false );

		/**
		 * Searches for child by id.
		 *
		 * @param child id being searched
		 * @param recursive whether to search recursively
		 * @returns nullptr if not found, or *first* child otherwise (breadth first)
		 */
		object* find( string child, bool recursive = false );

		/**
		 * Returns the parent of this object.
		 *
		 * If no parent is present, nullptr will be returned.
		 *
		 * @returns parent object of current object
		 */
		virtual object* get_parent() const { return m_parent; }

		/**
		 * Returns object UID
		 */
		uid get_uid() const { return m_uid; }

		/**
		 * Returns object ID
		 */
		const std::string& get_id() const { return m_id; }

		/**
		 * Returns object UID
		 */
		int get_lua_index() const { return m_lua_index; }

		/**
		 * Moves object to back of child list (top of stack)
		 *
		 * @returns true if object found, false otherwise
		 */
		bool move_to_top( object* child );

		/**
		 * Moves object to front of child list (bottom of stack)
		 *
		 * @returns true if object found, false otherwise
		 */
		bool move_to_bottom( object* child );

		list::iterator begin() { return m_children.begin(); }
		list::iterator end()   { return m_children.end(); }

		/**
		 * Destructor.
		 *
		 * Destroys all children, unregisters the object from the uid_store.
		 */
	protected:
		virtual ~object();

	protected:
		string  m_id;              ///< id type of the object
		string  m_name;            ///< name of the object
		uid     m_uid;             ///< uid of the object
		int     m_lua_index;       ///< lua reference
		int     m_lua_proto_index; ///< lua reference
		object* m_parent;          ///< pointer to parent
		list    m_children;        ///< children objects
		size_t  m_child_count;     ///< number of children
	};

} // namespace nv

#endif // NV_OBJECT_HH
