// Copyright (C) 2012-2013 ChaosForge / Kornel Kisielewicz
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

#ifndef NV_LUA_VALUES_HH
#define NV_LUA_VALUES_HH

#include <nv/common.hh>
#include <nv/type_traits.hh>
#include <nv/string.hh>

struct lua_State;

namespace nv
{
	namespace lua
	{
		typedef ptrdiff_t     linteger;
		typedef unsigned long lunsigned;
		typedef double        lnumber;

 		struct passer
		{
			virtual void push( lua_State *L ) const = 0;
			virtual ~passer(){}
		};

		template < typename T >
		struct returner
		{
			returner( lua_State *, int ) : value() {}
			returner( lua_State *, int, const T& ) : value() {}
			operator T() { return value; }
			operator const T() const { return value; }
			virtual returner<T> to( lua_State *L, int index ) = 0;
			virtual returner<T> to( lua_State *L, int index, const T& def ) = 0;
			virtual ~returner(){}
		protected:
			T value;
		};

		template < typename T >
		struct pass_traits
		{
			static void push( lua_State *L, const T& p ) { p.push( L ); }
			static T to( lua_State *L, int index ) { return T( L, index ); }
			static T to( lua_State *L, int index, const T& def ) { return T( L, index, def ); }
		};

		namespace detail
		{
			int upvalue_index( int i );

			void push_unsigned( lua_State *L, lunsigned v );
			void push_integer ( lua_State *L, linteger v );
			void push_number  ( lua_State *L, lnumber v );
			void push_bool    ( lua_State *L, bool v );
			void push_string  ( lua_State *L, const std::string& s );
			void push_cstring ( lua_State *L, const char* s );
			void push_object  ( lua_State *L, object* o );
			void push_pointer ( lua_State *L, void* p );

			lunsigned   to_unsigned( lua_State *L, int index );
			linteger    to_integer ( lua_State *L, int index );
			lnumber     to_number  ( lua_State *L, int index );
			bool        to_bool    ( lua_State *L, int index );
			std::string to_string  ( lua_State *L, int index );
			const char* to_cstring ( lua_State *L, int index );
			object*     to_object  ( lua_State *L, int index );
			void*       to_pointer ( lua_State *L, int index );

			lunsigned   to_unsigned( lua_State *L, int index, lunsigned def );
			linteger    to_integer ( lua_State *L, int index, linteger def );
			lnumber     to_number  ( lua_State *L, int index, lnumber def );
			bool        to_bool    ( lua_State *L, int index, bool def );
			std::string to_string  ( lua_State *L, int index, const std::string& def );
			const char* to_cstring ( lua_State *L, int index, const char* def );
			object*     to_object  ( lua_State *L, int index, object* def );
			void*       to_pointer ( lua_State *L, int index, void* def );

			void pop_and_discard( lua_State *L, int count );
			bool is_userdata( lua_State *L, int index, const char* metatable );
			void* check_userdata( lua_State *L, int index, const char* metatable );
			void* allocate_userdata( lua_State *L, size_t size, const char* metatable );
			template <typename T>
			void push_userdata( lua_State *L, const T& v, const char* metatable )
			{
				new (allocate_userdata(L, sizeof(T), metatable)) T(v);
			}
		}

		template <>
		struct pass_traits<linteger>
		{
			static void push( lua_State *L, linteger s ) { detail::push_integer( L, s ); }
			static linteger to( lua_State *L, int index ) { return detail::to_integer( L, index ); }
			static linteger to( lua_State *L, int index, linteger def ) { return detail::to_integer( L, index, def ); }
		};

		template <>
		struct pass_traits<lunsigned>
		{
			static void push( lua_State *L, lunsigned s ) { detail::push_unsigned( L, s ); }
			static lunsigned to( lua_State *L, int index ) { return detail::to_unsigned( L, index ); }
			static lunsigned to( lua_State *L, int index, lunsigned def ) { return detail::to_unsigned( L, index, def ); }
		};

		template <>
		struct pass_traits<lnumber>
		{
			static void push( lua_State *L, lnumber s ) { detail::push_number( L, s ); }
			static lnumber to( lua_State *L, int index ) { return detail::to_number( L, index ); }
			static lnumber to( lua_State *L, int index, lnumber def ) { return detail::to_number( L, index, def ); }
		};

		template <>
		struct pass_traits<bool>
		{
			static void push( lua_State *L, bool s ) { detail::push_bool( L, s ); }
			static bool to( lua_State *L, int index ) { return detail::to_bool( L, index ); }
			static bool to( lua_State *L, int index, bool def ) { return detail::to_bool( L, index, def ); }
		};

		template <>
		struct pass_traits<const char*>
		{
			static void push( lua_State *L, const char* s ) { detail::push_cstring( L, s ); }
			static const char* to( lua_State *L, int index ) { return detail::to_cstring( L, index ); }
			static const char* to( lua_State *L, int index, const char* def ) { return detail::to_cstring( L, index, def ); }
		};

		template <>
		struct pass_traits<std::string>
		{
			static void push( lua_State *L, const std::string& s ) { detail::push_string( L, s ); }
			static std::string to( lua_State *L, int index ) { return detail::to_string( L, index ); }
			static std::string to( lua_State *L, int index, const std::string& def ) { return detail::to_string( L, index, def ); }
		};

		template <>
		struct pass_traits<object*>
		{
			static void push( lua_State *L, object* o ) { detail::push_object( L, o ); }
			static object* to( lua_State *L, int index ) { return detail::to_object( L, index ); }
			static object* to( lua_State *L, int index, object* def ) { return detail::to_object( L, index, def ); }
		};

		namespace detail
		{

			template <typename T, class ENABLE = void >
			struct type_degrade
			{
				typedef T type;
			};

			template <typename T>
			struct type_degrade< T, typename std::enable_if< std::is_floating_point< T >::value >::type >
			{
				typedef lnumber type;
			};

			template <typename T>
			struct type_degrade< T, typename std::enable_if< std::is_integral< T >::value && std::is_signed< T >::value >::type >
			{
				typedef linteger type;
			};

			template <typename T>
			struct type_degrade< T, typename std::enable_if< std::is_integral< T >::value && std::is_unsigned< T >::value >::type >
			{
				typedef lunsigned type;
			};

			template <typename T>
			struct type_degrade< T, typename std::enable_if< is_cstring< T >::value >::type >
			{
				typedef const char* type;
			};

			template <typename T>
			struct type_degrade< T, typename std::enable_if< is_stdstring< T >::value >::type >
			{
				typedef std::string type;
			};

			template <typename T>
			struct type_degrade< T*, typename std::enable_if< std::is_base_of<object, T >::value >::type >
			{
				typedef object* type;
			};

			template <>
			struct type_degrade< bool >
			{
				typedef bool type;
			};

 			template < typename T >
 			void push_value( lua_State *L, const T& p )
 			{
				typedef typename type_degrade<T>::type degraded;
 				pass_traits<degraded>::push( L, p );
 			}

			template < typename T1 >
			void push_values( lua_State *L, const T1& p1 ) { push_value( L, p1 ); }
			template < typename T1, typename T2 >
			void push_values( lua_State *L, const T1& p1, const T2& p2 ) { push_value( L, p1 ); push_value( L, p2 ); }
			template < typename T1, typename T2, typename T3 >
			void push_values( lua_State *L, const T1& p1, const T2& p2, const T3& p3 ) { push_value( L, p1 ); push_value( L, p2 ); push_value( L, p3 ); }
			template < typename T1, typename T2, typename T3, typename T4 >
			void push_values( lua_State *L, const T1& p1, const T2& p2, const T3& p3, const T4& p4 ) { push_value( L, p1 ); push_value( L, p2 ); push_value( L, p3 ); push_value( L, p4 ); }
			template < typename T1, typename T2, typename T3, typename T4, typename T5 >
			void push_values( lua_State *L, const T1& p1, const T2& p2, const T3& p3, const T4& p4, const T5& p5 ) { push_value( L, p1 ); push_value( L, p2 ); push_value( L, p3 ); push_value( L, p4 ); push_value( L, p5 ); }

			template < typename T >
			inline void pop_value( lua_State *L, T& p )
			{
				typedef typename type_degrade<T>::type degraded;
				p = (T)pass_traits<degraded>::to( L, -1 );
				detail::pop_and_discard(L, 1);
			}
			template < typename T >
			inline T pop_return_value( lua_State *L )
			{
				typedef typename type_degrade<T>::type degraded;
				T ret;
				ret = (T)pass_traits<degraded>::to( L, -1 );
				detail::pop_and_discard(L, 1);
				return ret;
			}

			template <>
			inline void pop_return_value<void>( lua_State* )
			{
			}

			template < typename T >
			inline typename std::remove_reference<T>::type get_value( lua_State *L, int index )
			{
				typedef typename type_degrade<T>::type degraded;
				return pass_traits<degraded>::to( L, index );
			}

			template < typename T >
			inline T get_value( lua_State *L, int index, const T& def )
			{
				typedef typename type_degrade<T>::type degraded;
				return pass_traits<degraded>::to( L, index, def );
			}

			template < typename T >
			struct return_count
			{
				const static int value = 1;
			};

			template <>
			struct return_count<void>
			{
				const static int value = 0;
			};

		}

	} // namespace lua

} // namespace nv

#endif // NV_LUA_VALUES_HH
