// Copyright (C) 2012-2013 ChaosForge / Kornel Kisielewicz
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

/**
 * @file flags.hh
 * @author Kornel Kisielewicz
 * @brief flags
 */

#ifndef NV_FLAGS_HH
#define NV_FLAGS_HH

#include <nv/common.hh>
#include <nv/type_traits.hh>
#include <bitset>

namespace nv
{
	template< typename T, typename IS_ENUM >
	struct base_underlying_type_helper
	{
		typedef T type;
	};

	template< typename T >
	struct base_underlying_type_helper< T, std::true_type >
	{
		typedef typename nv::underlying_type<T>::type type;
	};


	template< typename T >
	struct base_underlying_type
	{
		typedef typename base_underlying_type_helper< T, typename std::is_enum<T>::type >::type type;
	};

	template < uint32 SIZE, typename T = uint32 >
	class flags
	{
	public:

		class reference
		{
			friend class flags<SIZE,T>;
		public:
			typedef bool value_type;
			typedef T    index_type;

			reference& operator=( value_type a_value )
			{
				m_flags->set( m_index, a_value );
				return (*this);
			}
			reference& operator=( const reference& a_value )
			{
				m_flags->set( m_index, bool( a_value ) );
				return (*this);
			}
			operator bool() const 
			{
				return m_flags->test( m_index );
			}

		private:
			reference() : m_flags( nullptr ), m_index( 0 ) {}

			reference( flags<SIZE,T>* a_flags, index_type a_index )
				: m_flags( a_flags ), m_index( a_index )
			{	
			}

		private:
			flags<SIZE,T>* m_flags;
			index_type     m_index;
		};

		typedef uint8  data_type;
		typedef bool   value_type;
		typedef T      index_type;
		typedef uint32 size_type;
		typedef typename base_underlying_type<T>::type raw_index_type;

		static const size_type data_type_size = sizeof( data_type ) * 8;
		static const size_type data_count     = SIZE;
		static const size_type data_size      = (SIZE+data_type_size-1) / data_type_size;

		flags()
		{
			reset();
		}

		explicit flags( const data_type* a_data )
		{
			assign( a_data );
		}

		void assign( const data_type* a_data )
		{
			std::copy( a_data, a_data + data_size, m_data );
		}

		void reset()
		{
			std::fill( m_data, m_data + data_size, 0 );
		}

		void include( index_type i )
		{
			raw_index_type idx = static_cast< raw_index_type >( i ) / data_type_size;
			raw_index_type pos = static_cast< raw_index_type >( i ) % data_type_size;
			m_data[ idx ] |= 1 << static_cast< data_type >( pos );
		}

		void exclude( index_type i )
		{
			raw_index_type idx = static_cast< raw_index_type >( i ) / data_type_size;
			raw_index_type pos = static_cast< raw_index_type >( i ) % data_type_size;
			m_data[ idx ] &= ~(1 << static_cast< data_type >( pos ) );
		}

		void set( index_type i, value_type value )
		{
			raw_index_type idx = static_cast< raw_index_type >( i ) / data_type_size;
			raw_index_type pos = static_cast< raw_index_type >( i ) % data_type_size;
			if ( value )
				m_data[ idx ] |= 1 << static_cast< data_type >( pos );
			else
				m_data[ idx ] &= ~(1 << static_cast< data_type >( pos ) );
		}

		bool test( index_type i ) const
		{
			raw_index_type idx = static_cast< raw_index_type >( i ) / data_type_size;
			raw_index_type pos = static_cast< raw_index_type >( i ) % data_type_size;
			return ( m_data[ idx ] & ( 1 << static_cast< data_type >( pos ) ) ) != 0;
		}

		const data_type* data() const
		{
			return m_data;
		}

		data_type* data()
		{
			return m_data;
		}

		size_type size() const
		{
			return data_count;
		}

		size_type capacity() const
		{
			return data_size;
		}

		bool operator[]( index_type idx ) const
		{
			return test( idx );
		}

		reference operator[]( index_type idx )
		{
			return reference( this, idx );
		}

	private:
		data_type m_data[ data_size ];
	};

} // namespace nv

#endif // NV_FLAGS_HH
