// Copyright (C) 2012-2013 ChaosForge / Kornel Kisielewicz
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh
/**
 * @file uniform.hh
 * @author Kornel Kisielewicz epyon@chaosforge.org
 * @brief uniform classes
 */

#ifndef NV_UNIFORM_HH
#define NV_UNIFORM_HH

#include <unordered_map>
#include <nv/interface/camera.hh>
#include <nv/common.hh>
#include <nv/string.hh>

namespace nv
{
	class context;

	class uniform_base
	{
	public: 
		uniform_base( const string& name, datatype type, int location, int length ) 
			: m_name( name ), m_type( type ), m_location(location), m_length( length ), m_dirty( true ) {}
		datatype get_type() const { return m_type; }
		int get_location() const { return m_location; }
		int get_length() const { return m_length; }
		bool is_dirty() const { return m_dirty; }
		void clean() { m_dirty = false; }
		virtual ~uniform_base() {}
	protected:
		string   m_name;
		datatype m_type;
		int      m_location;
		int      m_length;
		bool     m_dirty;
	};

	template< typename T >
	class uniform : public uniform_base
	{
	public:
		typedef T value_type;

		uniform( const string& name, int location, int length ) 
			: uniform_base( name, type_to_enum< T >::type, location, length ), m_value( nullptr )
		{
			m_value = new T[ m_length ];
		}

		void set_value( const T& value ) 
		{ 
			NV_ASSERT( m_length == 1, "set_value on array uniform!" );
			if ( value != m_value[0] )
			{
				m_value[0] = value; 
				m_dirty = true;
			}
		}

		void set_value( const T* value, uint32 count ) 
		{ 
			// TODO: memcmp?
			std::copy( value, value + count, m_value );
			m_dirty = true;
		}

		const T* get_value() { return m_value; }
		virtual ~uniform() 
		{
			delete[] m_value;
		}
	protected:
		T* m_value;
	};

	class engine_uniform_base
	{
	public:
		virtual void set( const context* , const scene_state* ) = 0;
		virtual ~engine_uniform_base() {}
	};

	template < typename T >
	class engine_uniform : public engine_uniform_base
	{
	public:
		typedef T          value_type;
		typedef uniform<T> uniform_type;

		engine_uniform( uniform_base* u ) : m_uniform( (uniform<T>*)u ) {}
	protected:
		uniform<T>* m_uniform;
	};

	class engine_uniform_factory_base
	{
	public:
		virtual engine_uniform_base* create( uniform_base* ) = 0;
		virtual ~engine_uniform_factory_base() {}
	};

	template < typename T >
	class engine_uniform_factory : public engine_uniform_factory_base
	{
	public:
		virtual engine_uniform_base* create( uniform_base* u )
		{
			return new T( u );
		}
	};

	class engine_link_uniform_base
	{
	public:
		virtual void set( uniform_base* ) = 0;
		virtual ~engine_link_uniform_base() {}
	};

	template < typename T >
	class engine_link_uniform : public engine_link_uniform_base
	{
	public:
		typedef T          value_type;
		typedef uniform<T> uniform_type;

		engine_link_uniform() {}
		virtual void set( uniform_base* u ) { set_impl( (uniform<T>*)u ); }
		virtual void set_impl( uniform<T>* u ) = 0;
	};

	typedef std::unordered_map< string, uniform_base* >                uniform_map;
	typedef std::vector< engine_uniform_base* >                        engine_uniform_list;
	typedef std::unordered_map< string, engine_uniform_factory_base* > engine_uniform_factory_map;
	typedef std::unordered_map< string, engine_link_uniform_base* >    engine_link_uniform_factory_map;

	class engine_uniform_m_view : public engine_uniform< mat4 >
	{
	public:
		engine_uniform_m_view( uniform_base* u ) : engine_uniform( u ) {}
		virtual void set( const context* , const scene_state* s ) { m_uniform->set_value( s->get_view() ); }
	};

	class engine_uniform_m_view_inv : public engine_uniform< mat4 >
	{
	public:
		engine_uniform_m_view_inv( uniform_base* u ) : engine_uniform( u ) {}
		virtual void set( const context* , const scene_state* s ) { m_uniform->set_value( s->get_view_inv() ); }
	};

	class engine_uniform_m_model : public engine_uniform< mat4 >
	{
	public:
		engine_uniform_m_model( uniform_base* u ) : engine_uniform( u ) {}
		virtual void set( const context* , const scene_state* s ) { m_uniform->set_value( s->get_model() ); }
	};

	class engine_uniform_m_model_inv : public engine_uniform< mat4 >
	{
	public:
		engine_uniform_m_model_inv( uniform_base* u ) : engine_uniform( u ) {}
		virtual void set( const context* , const scene_state* s ) { m_uniform->set_value( s->get_model_inv() ); }
	};

	class engine_uniform_m_modelview : public engine_uniform< mat4 >
	{
	public:
		engine_uniform_m_modelview( uniform_base* u ) : engine_uniform( u ) {}
		virtual void set( const context* , const scene_state* s ) { m_uniform->set_value( s->get_modelview() ); }
	};

	class engine_uniform_m_projection : public engine_uniform< mat4 >
	{
	public:
		engine_uniform_m_projection( uniform_base* u ) : engine_uniform( u ) {}
		virtual void set( const context* , const scene_state* s ) { m_uniform->set_value( s->get_projection() ); }
	};

	class engine_uniform_m_normal : public engine_uniform< mat3 >
	{
	public:
		engine_uniform_m_normal( uniform_base* u ) : engine_uniform( u ) {}
		virtual void set( const context* , const scene_state* s ) { m_uniform->set_value( s->get_normal() ); }
	};

	class engine_uniform_m_mvp : public engine_uniform< mat4 >
	{
	public:
		engine_uniform_m_mvp( uniform_base* u ) : engine_uniform( u ) {}
		virtual void set( const context* , const scene_state* s ) { m_uniform->set_value( s->get_mvp() ); }
	};

	class engine_uniform_v_camera_position : public engine_uniform< vec3 >
	{
	public:
		engine_uniform_v_camera_position( uniform_base* u ) : engine_uniform( u ) {}
		virtual void set( const context* , const scene_state* s ) { m_uniform->set_value( s->get_camera().get_position() ); }
	};

	class engine_uniform_v_camera_direction : public engine_uniform< vec3 >
	{
	public:
		engine_uniform_v_camera_direction( uniform_base* u ) : engine_uniform( u ) {}
		virtual void set( const context* , const scene_state* s ) { m_uniform->set_value( s->get_camera().get_direction() ); }
	};

	template< int VALUE >
	class engine_link_uniform_int : public engine_link_uniform< int >
	{
	public:
		engine_link_uniform_int() {}
		virtual void set_impl( uniform<int>* u ) { u->set_value(VALUE); }
	};



}

#endif // NV_UNIFORM_HH
