// Copyright (C) 2012 Kornel Kisielewicz
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

#ifndef NV_STRING_HH
#define NV_STRING_HH

#include <string>
#include <sstream>
#include <fstream>
#include <nv/common.hh>
#include <nv/exception.hh>

namespace nv
{
	using std::string;

    /**
     * Utility function for converting any value to string.
     *
     * @param value value to be converted, needs to have << operator
     *        to stream
     * @throw ore::runtime_error Throws ore::runtime_error on conversion fail
     * @return value converted to string
     */
    template < class T >
    string to_string( const T& value )
    {
      std::stringstream stream;
      stream << value;

      if ( stream.fail() )
      {
        throw runtime_error( "bad cast" );
      }

      return stream.str();
    }

    /**
     * Override function for special treatment of strings. Returns the
     * value passed.
     */
    inline string to_string( const string& value)
    {
        return value;
    }

    /**
     * Utility function for converting a string to the given type
     *
     * @param vin the string representing the value
     * @param vout the value to be read. Must be streamable with >>
     * @throw ore::runtime_error Throws ore::runtime_error on conversion fail
     */
    template < class T >
    void from_string( const string& vin, T& vout )
    {
      std::istringstream value( vin );
      value >> vout;

      if ( value.fail() )
      {
        throw runtime_error( "bad cast" );
      }
    }

    /**
     * Utility function for converting a string to the given type
     *
     * @param vin the string representing the value
     * @throw ore::runtime_error Throws ore::runtime_error on conversion fail
     */
	template < class T >
	T string_cast( const string& vin )
	{
		T vout;
		std::istringstream value( vin );
		value >> vout;

		if ( value.fail() )
		{
			throw runtime_error( "bad cast" );
		}
		return vout;
	}


    /**
     * Override function for special treatment of strings. Returns the
     * value passed.
     */
    inline void from_string( const string& vin, string& vout )
	{
      vout = vin;
    }

	/**
     * Override function for special treatment of strings. Returns the
     * value passed.
     */
	template <>
    inline std::string string_cast( const string& vin )
	{
		return vin;
    }


	/**
	 * Simple function for slurping a file into a string.
	 */
	inline std::string slurp( const std::string& filename )
	{
		std::ifstream input(filename);
		if ( !input.is_open() ) throw std::runtime_error("File "+filename+" not found!");
		std::stringstream sstr;
		while(input >> sstr.rdbuf());
		return sstr.str();
	}


}

#endif // NV_STRING_HH
