// Copyright (C) 2014 ChaosForge / Kornel Kisielewicz
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

#ifndef NV_ANIMATION_HH
#define NV_ANIMATION_HH

#include <nv/common.hh>
#include <vector>
#include <nv/interface/stream.hh>
#include <nv/math.hh>
#include <nv/transform.hh>
#include <glm/gtc/matrix_transform.hpp>

namespace nv
{

	// TODO: time sorting or check?
	template < typename KEY >
	class key_vector
	{
	public:
		struct key
		{
			float time;
			KEY   value;
			key() {}
			key( float a_time, const KEY& a_value ) : time(a_time), value(a_value) {}
		};
		key_vector() {}
		void insert( float a_time, const KEY& a_key ) { m_keys.emplace_back( a_time, a_key ); }
		size_t size() const { return m_keys.size(); }
		const key* data() const { return m_keys.data(); }
		const KEY& get( size_t index ) const { return m_keys[index]; }
		KEY get_interpolated( float time ) const
		{
			if ( m_keys.size() == 0 ) return KEY();
			if ( m_keys.size() == 1 ) return m_keys[0].value;
			int index = -1;
			for ( int i = 0 ; i < (int)m_keys.size() - 1 ; i++ )
			{
				if ( time < m_keys[i + 1].time ) { index = i; break; }
			}
			NV_ASSERT( index >= 0, "animation time fail!");
			float delta  = m_keys[index + 1].time - m_keys[index].time;
			float factor = glm::clamp( (time - m_keys[index].time) / delta, 0.0f, 1.0f );
			return interpolate( m_keys[index].value, m_keys[index + 1].value, factor );
		}
		virtual void dump( stream* out_stream )
		{
			size_t sz = m_keys.size();
			out_stream->write( &sz, sizeof( size_t ), 1 );
			if ( sz > 0 )
			{
				out_stream->write( &m_keys[0], sizeof( key ), sz );
			}
		}
		virtual void load( stream* in_stream )
		{
			size_t sz;
			in_stream->read( &sz, sizeof( size_t ), 1 );
			if ( sz > 0 )
			{
				m_keys.resize( sz );
				in_stream->read( &m_keys[0], sizeof( key ), sz );
			}
		}
	protected:
		std::vector< key > m_keys;
	};

	class key_animation_data
	{
	public:
		virtual mat4 get_matrix( float time ) const = 0;
		virtual transform get_transform( float time ) const = 0;
		virtual void dump( stream* out_stream ) = 0;
		virtual void load( stream* in_stream ) = 0;
		virtual ~key_animation_data() {}
	};


	class key_vectors_prs : public key_animation_data
	{
	public:
		key_vectors_prs() {}
		void insert_position( float a_time, const vec3& a_value ) { m_positions.insert( a_time, a_value ); }
		void insert_rotation( float a_time, const quat& a_value ) { m_rotations.insert( a_time, a_value ); }
		void insert_scale   ( float a_time, const vec3& a_value ) { m_scales.insert( a_time, a_value ); }
		bool empty() const { return m_positions.size() == 0 && m_rotations.size() == 0 && m_scales.size() == 0; }
		virtual mat4 get_matrix( float time ) const
		{
			nv::mat4 position;
			nv::mat4 rotation;
			nv::mat4 scaling;

			if ( m_positions.size() > 0 ) position = glm::translate( position, m_positions.get_interpolated( time ) );
			if ( m_rotations.size() > 0 ) rotation = glm::mat4_cast( m_rotations.get_interpolated( time ) );
			if ( m_scales.size()    > 0 ) scaling  = glm::scale( scaling, m_scales.get_interpolated( time ) );

			return position * rotation * scaling;
		}
		virtual transform get_transform( float time ) const
		{
			transform result;
			if ( m_positions.size() > 0 ) result.set_position( m_positions.get_interpolated( time ) );
			if ( m_rotations.size() > 0 ) result.set_orientation( m_rotations.get_interpolated( time ) );
			return result;
		}
		virtual void dump( stream* out_stream )
		{
			m_positions.dump( out_stream );
			m_rotations.dump( out_stream );
			m_scales.dump( out_stream );
		}
		virtual void load( stream* in_stream )
		{
			m_positions.load( in_stream );
			m_rotations.load( in_stream );
			m_scales.load( in_stream );
		}
	protected:
		key_vector< nv::vec3 > m_positions;
		key_vector< nv::quat > m_rotations;
		key_vector< nv::vec3 > m_scales;
	};

	class transform_vector : public key_animation_data
	{
	public:
		transform_vector() {}
		void reserve( size_t sz ) { m_keys.reserve( sz ); }
		void insert( const transform& t ) { m_keys.push_back( t ); }
		size_t size() const { return m_keys.size(); }
		const transform& get( size_t index ) const { return m_keys[ index ]; }
		const transform* data() const { return m_keys.data(); }

		virtual void dump( stream* out_stream )
		{
			size_t sz = m_keys.size();
			out_stream->write( &sz, sizeof( size_t ), 1 );
			if ( sz > 0 )
			{
				out_stream->write( &m_keys[0], sizeof( transform ), sz );
			}
		}
		virtual void load( stream* in_stream )
		{
			size_t sz;
			in_stream->read( &sz, sizeof( size_t ), 1 );
			if ( sz > 0 )
			{
				m_keys.resize( sz );
				in_stream->read( &m_keys[0], sizeof( transform ), sz );
			}
		}
		virtual mat4 get_matrix( float time ) const
		{
			return get_transform( time ).extract();
		}
		virtual transform get_transform( float time ) const
		{
			if ( m_keys.size() == 0 ) return transform();
			if ( m_keys.size() == 1 ) return m_keys[0];
			size_t index = glm::clamp<size_t>( size_t( time ), 0, m_keys.size() - 2 );
			float factor = glm::clamp<float> ( time - index, 0.0f, 1.0f );
			return interpolate( m_keys[ index ], m_keys[ index + 1 ], factor );
		}
	protected:
		std::vector< transform > m_keys;
	};

}

#endif // NV_ANIMATION_HH
