// Copyright (C) 2012-2013 ChaosForge / Kornel Kisielewicz
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

#include "nv/lua/lua_values.hh"

#include "nv/lua/lua_raw.hh"
#include "nv/object.hh"

using nv::lua::linteger;
using nv::lua::lnumber;
using nv::lua::lunsigned;

int nv::lua::detail::upvalue_index( int i )
{
	return lua_upvalueindex(i);
}

bool nv::lua::detail::is_userdata( lua_State *L, int index, const char* metatable )
{
	return luaL_testudata( L, index, metatable ) != 0;
}

void* nv::lua::detail::check_userdata( lua_State *L, int index, const char* metatable )
{
	return luaL_checkudata( L, index, metatable );
}

void* nv::lua::detail::allocate_userdata( lua_State *L, size_t size, const char* metatable )
{
	void* result = lua_newuserdata(L, size);
	luaL_setmetatable( L, metatable );
	return result;
}

void nv::lua::detail::pop_and_discard( lua_State *L, int count )
{
	lua_pop( L, count );
}

void nv::lua::detail::push_unsigned( lua_State *L, lunsigned v )
{
	lua_pushunsigned( L, v );
}

void nv::lua::detail::push_integer ( lua_State *L, linteger v )
{
	lua_pushinteger( L, v );
}

void nv::lua::detail::push_number  ( lua_State *L, lnumber v )
{
	lua_pushnumber( L, v );
}

void nv::lua::detail::push_bool    ( lua_State *L, bool v )
{
	lua_pushboolean( L, v );
}

void nv::lua::detail::push_string  ( lua_State *L, const std::string& s )
{
	lua_pushlstring( L, s.c_str(), s.size() );
}

void nv::lua::detail::push_cstring ( lua_State *L, const char* s )
{
	lua_pushstring( L, s );
}

void nv::lua::detail::push_object  ( lua_State *L, object* o )
{
	if ( o == nullptr )
	{
		lua_pushnil( L );
	}
	else
	{
		lua_rawgeti( L, LUA_REGISTRYINDEX, o->get_lua_index() );
	}
}

void nv::lua::detail::push_pointer ( lua_State *L, void* p )
{
	lua_pushlightuserdata( L, p );
}

lunsigned   nv::lua::detail::to_unsigned( lua_State *L, int index )
{
	return lua_tounsigned( L, index );
}

linteger    nv::lua::detail::to_integer ( lua_State *L, int index )
{
	return lua_tointeger( L, index );
}

lnumber     nv::lua::detail::to_number  ( lua_State *L, int index )
{
	return lua_tonumber( L, index );
}

bool        nv::lua::detail::to_bool    ( lua_State *L, int index )
{
	return lua_toboolean( L, index ) != 0;
}

std::string nv::lua::detail::to_string  ( lua_State *L, int index )
{
	return lua_tostring( L, index );
}

const char* nv::lua::detail::to_cstring ( lua_State *L, int index )
{
	return lua_tostring( L, index );
}

nv::object* nv::lua::detail::to_object  ( lua_State *L, int index )
{
	object* o = nullptr;
	if ( lua_istable( L , index ) )
	{
		lua_pushstring( L, "__ptr" );
		lua_rawget( L, index );
		if ( lua_isuserdata( L, -1 ) )
		{
			o = static_cast<object*>( lua_touserdata( L, -1 ) );
		} 
		lua_pop( L, 1 );
	}
	return o;
}

void*       nv::lua::detail::to_pointer ( lua_State *L, int index )
{
	return lua_touserdata( L, index );
}

lunsigned   nv::lua::detail::to_unsigned( lua_State *L, int index, lunsigned def )
{
	return ( lua_type( L, index ) == LUA_TNUMBER ? lua_tounsigned( L, index ) : def );
}

linteger    nv::lua::detail::to_integer ( lua_State *L, int index, linteger def )
{
	return ( lua_type( L, index ) == LUA_TNUMBER ? lua_tointeger( L, index ) : def );
}

lnumber     nv::lua::detail::to_number  ( lua_State *L, int index, lnumber def )
{
	return ( lua_type( L, index ) == LUA_TNUMBER ? lua_tonumber( L, index ) : def );
}

bool        nv::lua::detail::to_bool    ( lua_State *L, int index, bool def )
{
	return ( lua_type( L, index ) == LUA_TBOOLEAN ? lua_toboolean( L, index ) != 0 : def );
}

std::string nv::lua::detail::to_string  ( lua_State *L, int index, const std::string& def )
{
	return ( lua_type( L, index ) == LUA_TSTRING ? lua_tostring( L, index ) : def );
}

const char* nv::lua::detail::to_cstring ( lua_State *L, int index, const char* def )
{
	return ( lua_type( L, index ) == LUA_TSTRING ? lua_tostring( L, index ) : def );
}

nv::object* nv::lua::detail::to_object  ( lua_State *L, int index, nv::object* def )
{
	object* o = def;
	if ( lua_istable( L , index ) )
	{
		lua_pushstring( L, "__ptr" );
		lua_rawget( L, index );
		if ( lua_isuserdata( L, -1 ) )
		{
			o = static_cast<object*>( lua_touserdata( L, -1 ) );
		}
		lua_pop( L, 1 );
	}
	return o;
}

void*       nv::lua::detail::to_pointer ( lua_State *L, int index, void* def )
{
	return ( lua_type( L, index ) == LUA_TUSERDATA ? lua_touserdata( L, index ) : def );
}