// Copyright (C) 2012-2013 ChaosForge / Kornel Kisielewicz
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

#ifndef NV_MESH_DATA_HH
#define NV_MESH_DATA_HH

#include <nv/common.hh>
#include <unordered_map>
#include <vector>
#include <nv/math.hh>
#include <nv/transform.hh>
#include <nv/interface/device.hh>

namespace nv
{
	class mesh_data
	{
	public:
		friend class mesh_data_creator;
		typedef std::vector< transform > transforms;
		typedef std::unordered_map< std::string, transforms > tag_map;

		mesh_data() {}

		const std::vector< vec3 >&   get_positions() const { return m_positions; }
		const std::vector< vec3 >&   get_normals()   const { return m_normals; }
		const std::vector< vec3 >&   get_tangents()  const { return m_tangents; }
		const std::vector< vec2 >&   get_texcoords() const { return m_texcoords; }
		const std::vector< uint32 >& get_indices()   const { return m_indices; }

		const mesh_data::tag_map&    get_tag_map()   const { return m_tags; }

		size_t get_frame_count() const  { return m_texcoords.size() == 0 ? 0 : m_positions.size() / m_texcoords.size(); }
		size_t get_index_count() const  { return m_indices.size(); }
		size_t get_vertex_count() const { return m_texcoords.size(); }

		const vec3* get_position_frame( uint32 frame = 0 ) { return m_positions.data() + m_texcoords.size() * frame; }
		const vec3* get_normal_frame  ( uint32 frame = 0 ) { return m_normals.data() + m_texcoords.size() * frame; }
		const vec3* get_tangent_frame ( uint32 frame = 0 ) { return m_tangents.data() + m_texcoords.size() * frame; }

		bool has_tags()     const { return m_tags.size() > 0;  }
		bool has_frames()   const { return get_frame_count() > 1;  }
		bool has_normals()  const { return m_normals.size() > 0;  }
		bool has_tangents() const { return m_tangents.size() > 0; }

	protected:
		tag_map               m_tags;
		std::vector< vec3 >   m_normals;
		std::vector< vec3 >   m_positions;
		std::vector< vec3 >   m_tangents;
		std::vector< vec2 >   m_texcoords;
		std::vector< uint32 > m_indices;
	};

	class mesh_data_creator
	{
	public:
		mesh_data_creator() 
		{
			m_data = new mesh_data;
		}

		const std::vector< vec3 >&   get_positions() const { return m_data->m_positions; }
		const std::vector< vec3 >&   get_normals()   const { return m_data->m_normals; }
		const std::vector< vec3 >&   get_tangents()  const { return m_data->m_tangents; }
		const std::vector< vec2 >&   get_texcoords() const { return m_data->m_texcoords; }
		const std::vector< uint32 >& get_indices()   const { return m_data->m_indices; }
		const mesh_data::tag_map&    get_tag_map()   const { return m_data->m_tags; }

		std::vector< vec3 >&   get_positions() { return m_data->m_positions; }
		std::vector< vec3 >&   get_normals()   { return m_data->m_normals; }
		std::vector< vec3 >&   get_tangents()  { return m_data->m_tangents; }
		std::vector< vec2 >&   get_texcoords() { return m_data->m_texcoords; }
		std::vector< uint32 >& get_indices()   { return m_data->m_indices; }
		mesh_data::tag_map&    get_tag_map()   { return m_data->m_tags; }

		mesh_data* release()
		{
			mesh_data* result = m_data;
			m_data = nullptr;
			return result;
		}

		~mesh_data_creator()
		{
			if ( m_data ) delete m_data;
		}
	private:
		mesh_data* m_data;
	};

}

#endif // NV_MESH_DATA_HH
