// Copyright (C) 2012-2013 ChaosForge / Kornel Kisielewicz
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

/**
 * @file md3_loader.hh
 * @author Kornel Kisielewicz
 * @brief md3 loader
 */

#ifndef NV_MD3_LOADER_HH
#define NV_MD3_LOADER_HH

#include <unordered_map>
#include <vector>
#include <nv/common.hh>
#include <nv/interface/mesh_loader.hh>

namespace nv 
{

	struct md3_tag
	{
		std::string name;
		mat4 transform;
	};

	class keyframed_loader : public mesh_loader
	{
	public:
		virtual size_t get_max_frames() const = 0;
		virtual void load_positions( std::vector<vec3>& p, sint32 frame =-1 ) = 0;
		virtual void load_normals( std::vector<vec3>& n, sint32 frame =-1 ) = 0;
		virtual void load_texcoords( std::vector<vec2>& t ) = 0;
		virtual void load_indicies( std::vector<uint16>& idx ) = 0;
	};


	class md3_loader : public keyframed_loader
	{
	public:
		md3_loader();
		virtual ~md3_loader();
		virtual bool load( stream& source );
		virtual size_t get_size() { return m_size; }
		virtual mesh* release_mesh();
		virtual mesh* get_frame( sint32 frame );
		virtual const md3_tag* get_tag( const std::string& name ) const;
		virtual mat4 get_tag( sint32 frame, const std::string& name ) const;
		size_t get_max_frames() const;
		void load_tag_names( std::vector< std::string >& tags );
		void load_tags( std::vector<mat4>& t, const std::string& tag );
		void load_positions( std::vector<vec3>& p, sint32 frame =-1 );
		void load_normals( std::vector<vec3>& n, sint32 frame =-1 );
		void load_texcoords( std::vector<vec2>& t );
		void load_indicies( std::vector<uint16>& idx );
	private:
		void* m_md3;
		std::unordered_map< std::string, md3_tag > m_tags;
		std::size_t m_size;
	};

	class keyframed_mesh_data
	{
	public:
		typedef std::vector< mat4 > transforms;
		typedef std::unordered_map< std::string, transforms > tag_map;

		keyframed_mesh_data( keyframed_loader* loader );
		size_t get_frame_count() const { return m_frames; }
		size_t get_index_count() const { return m_indices.size(); }
		size_t get_vertex_count() const { return m_texcoords.size(); }
		const std::vector< vec3 >& get_positions() { return m_positions; }
		const std::vector< vec3 >& get_normals() { return m_normals; }
		const std::vector< vec2 >& get_texcoords() { return m_texcoords; }
		const std::vector< uint16 >& get_indices() { return m_indices; }
		const tag_map& get_tag_map() { return m_tags; }
		const vec3* get_position_data( uint32 frame = 0 ) { return m_positions.data() + m_texcoords.size() * frame; }
		const vec3* get_normal_data( uint32 frame = 0 ) { return m_normals.data() + m_texcoords.size() * frame; }
	protected:

		std::size_t           m_frames;
		tag_map               m_tags;
		std::vector< vec3 >   m_normals;
		std::vector< vec3 >   m_positions;
		std::vector< vec2 >   m_texcoords;
		std::vector< uint16 > m_indices;
	};

}

#endif // NV_OBJ_LOADER_HH
