// Copyright (C) 2012 Kornel Kisielewicz
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

#ifndef NV_STRING_HH
#define NV_STRING_HH

#include <string>
#include <cstring>
#include <sstream>
#include <fstream>
#include <nv/common.hh>
#include <nv/exception.hh>

namespace nv
{
	using std::string;

	/**
	* Utility function for converting any value to string.
	*
	* @param value value to be converted, needs to have << operator
	*        to stream
	* @throw runtime_error Throws runtime_error on conversion fail
	* @return value converted to string
	*/
	template < class T >
	string to_string( const T& value )
	{
		std::stringstream stream;
		stream << value;

		if ( stream.fail() )
		{
			NV_THROW( runtime_error, "bad cast" );
		}

		return stream.str();
	}

	/**
	* Override function for special treatment of strings. Returns the
	* value passed.
	*/
	inline string to_string( const string& value)
	{
		return value;
	}

	/**
	* Utility function for converting a string to the given type
	*
	* @param vin the string representing the value
	* @param vout the value to be read. Must be streamable with >>
	* @throw runtime_error Throws runtime_error on conversion fail
	*/
	template < class T >
	void from_string( const string& vin, T& vout )
	{
		std::istringstream value( vin );
		value >> vout;

		if ( value.fail() )
		{
			NV_THROW( runtime_error, "bad cast" );
		}
	}

	/**
	* Utility function for converting a string to the given type
	*
	* @param vin the string representing the value
	* @throw runtime_error Throws runtime_error on conversion fail
	*/
	template < class T >
	T string_cast( const string& vin )
	{
		T vout;
		std::istringstream value( vin );
		value >> vout;

		if ( value.fail() )
		{
			NV_THROW( runtime_error, "bad cast" );
		}
		return vout;
	}


	/**
	* Override function for special treatment of strings. Returns the
	* value passed.
	*/
	inline void from_string( const string& vin, string& vout )
	{
		vout = vin;
	}

	/**
	* Override function for special treatment of strings. Returns the
	* value passed.
	*/
	template <>
	inline std::string string_cast( const string& vin )
	{
		return vin;
	}


	/**
	* Simple function for slurping a file into a string.
	*/
	inline std::string slurp( const std::string& filename )
	{
		std::ifstream input(filename);
		if ( !input.is_open() ) NV_THROW( std::runtime_error, "File "+filename+" not found!");
		std::stringstream sstr;
		while(input >> sstr.rdbuf());
		return sstr.str();
	}

	inline bool ends_with( const std::string& s, const std::string & ending )
	{
		if ( s.length() >= ending.length() ) {
			return ( 0 == s.compare (s.length() - ending.length(), ending.length(), ending) );
		} else {
			return false;
		}
	}

	template< typename T >
	struct string_length
	{
		static size_t get( T ) { return 0; }
	};
	template< size_t S >
	struct string_length< const char[S] >
	{
		static size_t get( const char* ) { return S-1; }
	};
	template<>
	struct string_length< const char* >
	{
		static size_t get( const char* s ) { return std::strlen( s ); }
	};
	template<>
	struct string_length< std::string >
	{
		static size_t get( const std::string& s ) { return s.length(); }
	};


}

#endif // NV_STRING_HH
