// Copyright (C) 2012-2013 ChaosForge / Kornel Kisielewicz
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

/**
 * @file mesh.hh
 * @author Kornel Kisielewicz
 * @brief mesh
 */

#ifndef NV_MESH_HH
#define NV_MESH_HH

#include <nv/common.hh>
#include <nv/string.hh>
#include <nv/types.hh>
#include <nv/interface/context.hh>
#include <unordered_map>
#include <vector>

namespace nv
{
	class vertex_attribute_base
	{
	public:
		vertex_attribute_base( const string& name, datatype attrtype ) 
			: m_name( name ), m_type( attrtype ) {}
		const string& get_name() const { return m_name; }
		datatype get_type() const { return m_type; }
		virtual datatype get_base_type() const = 0;
		virtual size_t get_components() const = 0;
		virtual size_t get_count() const = 0;
		virtual size_t get_size() const = 0;
		virtual void* get_data() const = 0;
		virtual ~vertex_attribute_base() {}
	private:
		string m_name;
		datatype  m_type;
	};

	template< typename T >
	class vertex_attribute : public vertex_attribute_base
	{
	public:
		typedef T value_type;
		typedef typename datatype_traits<T>::base_type base_type;
		typedef std::vector<T> list;

		vertex_attribute( const string& name ) : vertex_attribute_base( name, type_to_enum<T>::type ) {}
		const list& get() const { return m_list; }
		list& get() { return m_list; }
		virtual datatype get_base_type() const { return type_to_enum<base_type>::type; }
		virtual size_t get_components() const { return datatype_traits<T>::size; }
		virtual size_t get_count() const { return m_list.size(); }
		virtual size_t get_size() const { return m_list.size() * sizeof(T); }
		virtual void* get_data() const { return (void*)m_list.data(); }
	private:
		list m_list;
	};


	class mesh
	{
	public:
		typedef std::unordered_map< std::string, vertex_attribute_base* > map;

		mesh( primitive p = TRIANGLES ) 
			: m_map(), m_indices(), m_primitive( p ) {}
		template <typename T>
		vertex_attribute<T>* add_attribute( const string& attr ) 
		{ 
			vertex_attribute<T>* result = new vertex_attribute<T>( attr ); 
			m_map[ attr ] = result;
			return result;
		}
		template <typename T>
		vertex_attribute<T>* add_indices() 
		{ 
			if ( m_indices ) delete m_indices; // error?
			vertex_attribute<T>* result = new vertex_attribute<T>(""); 
			m_indices = result; 
			return result;
		}

		vertex_attribute_base* get_attribute( const string& attr )
		{
			map::iterator i = m_map.find( attr );
			return i != m_map.end() ? i->second : nullptr;
		}

		template <typename T>
		vertex_attribute<T>* get_attribute( const string& attr )
		{
			map::iterator i = m_map.find( attr );
			if ( i != m_map.end() && i->second->get_type() == type_to_enum<T>::type ) 
			{
				return ((vertex_attribute<T>*)(i->second));
			}
			return nullptr;
		}

		const vertex_attribute_base* get_indices() const { return m_indices; }
		vertex_attribute_base* get_indices() { return m_indices; }

		template <typename T>
		vertex_attribute<T>* get_indices() 
		{
			return m_indices->get_type() == type_to_enum<T>() ? ((vertex_attribute<T>*)(m_indices)) : nullptr; 
		}

		const map& get_attributes() const { return m_map; }
		primitive get_primitive() const { return m_primitive; }
		bool has_indices() const { return m_indices != nullptr; }

		~mesh() 
		{
			delete m_indices;
			for ( auto& v : m_map )
			{
				delete v.second;
			}
		}
	private:
		map                    m_map;
		vertex_attribute_base* m_indices;
		primitive              m_primitive;
	};

}


#endif // NV_MESH_HH

