// Copyright (C) 2012-2013 ChaosForge / Kornel Kisielewicz
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

#include "nv/lua/lua_aux.hh"

#include <utility>
#include "nv/lua/lua_raw.hh"
#include "nv/random.hh"

static int nluaaux_table_copy( lua_State* L )
{
	luaL_checktype( L, 1, LUA_TTABLE );
	lua_settop( L, 1 );
	nlua_shallowcopy( L, 1 );
	return 1;
}

static int nluaaux_table_deepcopy( lua_State* L )
{
	luaL_checktype( L, 1, LUA_TTABLE );
	lua_settop( L, 1 );
	nlua_deepcopy( L, 1 );
	return 0;
}

static int nluaaux_table_icopy( lua_State* L )
{
	luaL_checktype( L, 1, LUA_TTABLE );
	lua_settop( L, 1 );
	nlua_shallowicopy( L, 1 );
	return 1;
}

static int nluaaux_table_merge( lua_State* L )
{
	luaL_checktype( L, 1, LUA_TTABLE );
	luaL_checktype( L, 2, LUA_TTABLE );
	lua_settop( L, 2 );
	nlua_shallowmerge( L, 1 );
	return 0;
}

static int nluaaux_table_reversed( lua_State* L )
{
	luaL_checktype( L, 1, LUA_TTABLE );
	lua_settop( L, 1 );
	nlua_pushreversed( L, 1 );
	return 1;
}

static int nluaaux_table_toset( lua_State* L )
{
	luaL_checktype( L, 1, LUA_TTABLE );
	lua_settop( L, 1 );
	nlua_toset( L, 1 );
	return 1;
}

static int nluaaux_table_tokeyset( lua_State* L )
{
	luaL_checktype( L, 1, LUA_TTABLE );
	lua_settop( L, 1 );
	nlua_tokeyset( L, 1 );
	return 1;
}

static const struct luaL_Reg nluaaux_table_aux_f [] = {
	{ "copy",      nluaaux_table_copy },
	{ "deepcopy",  nluaaux_table_deepcopy },
	{ "icopy",     nluaaux_table_icopy },
	{ "merge",     nluaaux_table_merge },
	{ "reversed",  nluaaux_table_reversed },
	{ "toset",     nluaaux_table_toset },
	{ "tokeyset",  nluaaux_table_tokeyset },
	{ NULL, NULL }
};

static int nluaaux_math_clamp( lua_State* L )
{
	double v   = luaL_checknumber( L,  1 );
	double min = luaL_optnumber( L, 2, 0 );
	double max = luaL_optnumber( L, 3, 1 );  
	if ( min > max ) luaL_argerror( L, 2, "min is larger than max!" );
	lua_pushnumber( L, v < min ? 2 : ( v > max ? 3 : 1 ) );
	return 1;
}

static int nluaaux_math_dieroll( lua_State* L )
{
	int dice  = luaL_checkinteger( L,  1 );
	int sides = luaL_checkinteger( L,  2 );
	if ( dice < 1 )  luaL_argerror( L, 1, "die count lower than 1!" );
	if ( sides < 1 ) luaL_argerror( L, 2, "side count lower than 1!" );
	lua_pushnumber( L, nv::random::get().dice( dice, sides ) );
	return 1;
}

static int nluaaux_math_random( lua_State* L )
{
	if ( lua_gettop( L ) == 0 )
	{
		lua_pushnumber( L, nv::random::get().frand(1.0f) );
	}
	else
	{
		if ( lua_gettop( L ) == 1 )
		{
			int arg1 = luaL_checkinteger( L, 1 );
			if ( arg1 < 1 ) arg1 = 1;
			lua_pushunsigned( L, nv::random::get().urange( 1, arg1 ) );
		}
		else
		{
			int arg1 = luaL_checkinteger( L, 1 );
			int arg2 = luaL_checkinteger( L, 2 );
			if (arg2 < arg1) std::swap( arg2, arg1 );
			lua_pushinteger( L, nv::random::get().srange( arg1, arg2 ) );
		}
	}
	return 1;
}

static int nluaaux_math_randomseed( lua_State* L )
{
	nv::random::get().set_seed( static_cast< nv::uint32 > ( L, 1 ) );
	return 0;
}

static const struct luaL_Reg nluaaux_math_aux_f [] = {
	{ "clamp",     nluaaux_math_clamp },
	{ "random",    nluaaux_math_random },
	{ "randomseed",nluaaux_math_randomseed },
	{ "dieroll",   nluaaux_math_dieroll },
	{ NULL, NULL }
};

void nlua_register_aux( lua_State* L )
{
	nlua_register( L, "table", nluaaux_table_aux_f );
	nlua_register( L, "math", nluaaux_math_aux_f );
}

