// Copyright (C) 2012-2013 ChaosForge / Kornel Kisielewicz
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

#ifndef NV_COMMON_HH
#define NV_COMMON_HH

// NV Library version
#define NV_VERSION_MAJOR 0
#define NV_VERSION_MINOR 1
#define NV_VERSION_PATCH 0
#define NV_VERSION    (NV_VERSION_MAJOR << 16) | (NV_VERSION_MINOR << 8) | NV_VERSION_PATCH)

// Platform
#define NV_WINDOWS        1
#define NV_LINUX          2
#define NV_APPLE          3

// Compiler
#define NV_MSVC           1
#define NV_GNUC           2
#define NV_CLANG          3

// Endianess
#define NV_LITTLEENDIAN   1
#define NV_BIGENDIAN      2

// Bits
#define NV_32BIT          1
#define NV_64BIT          2

// Assumption
#define NV_ENDIANESS NV_LITTLEENDIAN

// Platform detection
#if defined( __WIN32__ ) || defined( _WIN32 )
#define NV_PLATFORM NV_WINDOWS
#elif defined( __APPLE_CC__)
#define NV_PLATFORM NV_APPLE
#else
#define NV_PLATFORM NV_LINUX
#endif

// Compiler detection
#if defined( _MSC_VER )
#define NV_COMPILER NV_MSVC
#define NV_COMP_VER _MSC_VER
#elif defined( __clang__ )
#define NV_COMPILER NV_CLANG
#define NV_COMP_VER (((__clang_major__)*100) + (__clang_minor__*10) + __clang_patchlevel__)
#elif defined( __GNUC__ )
#define NV_COMPILER NV_GNUC
#define NV_COMP_VER (((__GNUC__)*100) + (__GNUC_MINOR__*10) + __GNUC_PATCHLEVEL__)
#else
#error "Unknown compiler!"
#endif

// Architecture detection
#if defined(__x86_64__) || defined(_M_X64) || defined(__powerpc64__) || defined(__alpha__)
#define NV_ARCHITECTURE NV_64BIT
#elif defined(__ia64__) || defined(__s390__) || defined(__s390x__)
#define NV_ARCHITECTURE NV_64BIT
#else
#define NV_ARCHITECTURE NV_32BIT
#endif

// Platform specific settings.
#if NV_PLATFORM == NV_WINDOWS
#ifdef _DEBUG
#define NV_DEBUG 1
#else
#define NV_DEBUG 0
#endif
#endif

#if NV_PLATFORM == NV_LINUX || NV_PLATFORM == NV_APPLE
#ifdef DEBUG
#define NV_DEBUG 1
#else
#define NV_DEBUG 0
#endif
#endif

#if NV_PLATFORM == NV_LINUX || NV_PLATFORM == NV_APPLE
#define NV_POSIX
#endif

#if NV_COMPILER == NV_MSVC && NV_COMP_VER < 1600
#error "MSVC 2012+ required!"
#endif

#if NV_COMPILER == NV_GNUC && NV_COMP_VER < 460
#error "GCC 4.6+ required!"
#endif

#if NV_COMPILER == NV_CLANG && NV_COMP_VER < 320
#error "clang 3.2+ required!"
#endif

#if NV_COMPILER == NV_MSVC 
#pragma warning(disable: 4201)
#undef _SCL_SECURE_NO_WARNINGS // to prevent redefinig
#define _SCL_SECURE_NO_WARNINGS
#endif

#include <typeinfo>
#include <cstddef>
#include <cassert>
#include <nv/logging.hh>

#define NV_STRINGIZE_DETAIL(x) #x
#define NV_STRINGIZE(x) NV_STRINGIZE_DETAIL(x)

#if NV_COMPILER == NV_MSVC
#define NV_DEPRECATED(func) __declspec(deprecated) func
#elif NV_COMPILER == NV_GNUC || NV_COMPILER == NV_CLANG
#define NV_DEPRECATED(func) func __attribute__ ((deprecated))
#else 
#define NV_DEPRECATED(func) func
#endif 

#define NV_UNUSED(x) (void)(x)
#define NV_ASSERT(cond, msg) assert( (cond) && msg )
#define NV_THROW(eobj, ...) { \
	NV_LOG( nv::LOG_ERROR, __FILE__ " line " NV_STRINGIZE(__LINE__) " - exception thrown - " #eobj ); \
	throw eobj( __VA_ARGS__ ); \
} 

// MSVC and GCC is too stupid to notice fully covered enums, clang 
// is picky about it
#if NV_COMPILER == NV_CLANG
#define NV_RETURN_COVERED_DEFAULT( value ) 
#else
#define NV_RETURN_COVERED_DEFAULT( value ) default : return value
#endif

namespace nv
{
	class object;
	class root;
	class type_database;
	class uid_store;

	namespace lua
	{
		class state;
	}

	// Typedefs for fixed size types.
	typedef signed char         sint8;
	typedef signed short        sint16;
	typedef signed long         sint32;
	typedef signed long long    sint64;

	typedef unsigned char       uint8;
	typedef unsigned short      uint16;
	typedef unsigned long       uint32;
	typedef unsigned long long  uint64;

	typedef unsigned char       char8;
	typedef unsigned short      char16;
	typedef unsigned long       char32;

	typedef float  f32;
	typedef double f64;

	typedef uint64 uid;

} // namespace nv

template <typename OBJ, typename T> 
inline size_t offset_of(T OBJ::*ptr)
{
	return ((size_t)&(((OBJ*)0)->*ptr));
}

template <typename T, typename U>
inline T* down_cast(U* x)
{
#if NV_DEBUG
	T* p = dynamic_cast<T*>(x);
	if (p == 0)
	{
#ifdef NV_LOG
		NV_THROW( std::bad_cast );
#endif
	}

	return p;
#else
	return static_cast<T*>(x);
#endif
}

#endif // NV_COMMON_HH
