// Copyright (C) 2012-2013 Kornel Kisielewicz
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

#include "nv/gl/gl_device.hh"

#include "nv/gl/gl_window.hh"
#include "nv/gl/gl_program.hh"
#include "nv/gl/gl_vertex_buffer.hh"
#include "nv/gl/gl_texture2d.hh"
#include "nv/logging.hh"
#include "nv/lib/sdl.hh"
#include "nv/lib/sdl_image.hh"

using namespace nv;

window* gl_device::create_window( uint16 width, uint16 height )
{
	return new gl_window( this, width, height );
}

gl_device::gl_device()
{
	nv::load_sdl_library();
	m_info = NULL;

	if ( SDL_Init( SDL_INIT_VIDEO ) < 0 )
	{
		NV_LOG( LOG_CRITICAL, "Video initialization failed: " << SDL_GetError( ) );
		return; // TODO: Error report
	}

#if NV_SDL_VERSION == NV_SDL_12
	m_info = SDL_GetVideoInfo( );

	if ( !m_info )
	{
		NV_LOG( LOG_CRITICAL, "Video query failed: " << SDL_GetError( ) );
		return; // TODO: Error report
	}
#endif

//	bpp = m_info->vfmt->BitsPerPixel;

	SDL_GL_SetAttribute( SDL_GL_RED_SIZE, 8 );
	SDL_GL_SetAttribute( SDL_GL_GREEN_SIZE, 8 );
	SDL_GL_SetAttribute( SDL_GL_BLUE_SIZE, 8 );
	SDL_GL_SetAttribute( SDL_GL_DEPTH_SIZE, 24 );
	SDL_GL_SetAttribute( SDL_GL_DOUBLEBUFFER, 1 );

	SDL_GL_SetAttribute( SDL_GL_MULTISAMPLEBUFFERS, 1 );
	SDL_GL_SetAttribute( SDL_GL_MULTISAMPLESAMPLES, 4 );

#if NV_SDL_VERSION == NV_SDL_20
	SDL_GL_SetAttribute(SDL_GL_CONTEXT_MAJOR_VERSION, 2);
	SDL_GL_SetAttribute(SDL_GL_CONTEXT_MINOR_VERSION, 1);
	SDL_GL_SetAttribute(SDL_GL_DOUBLEBUFFER, 1);
	SDL_GL_SetAttribute(SDL_GL_DEPTH_SIZE, 24);
#endif

}

program* gl_device::create_program( const string& vs_source, const string& fs_source )
{
	return new gl_program( vs_source, fs_source );
}

vertex_buffer* gl_device::create_vertex_buffer( buffer_hint hint, size_t size, const void* source /*= nullptr */ )
{
	return new gl_vertex_buffer( hint, size, source );
}

index_buffer* gl_device::create_index_buffer( buffer_hint hint, size_t size, const void* source /*= nullptr */ )
{
	return new gl_index_buffer( hint, size, source );
}

vertex_array* gl_device::create_vertex_array()
{
	return new gl_vertex_array();
}

// this is a temporary function that will be removed once we find a way to 
// pass binary file data around
image_data* nv::gl_device::create_image_data( const std::string& filename )
{
	load_sdl_image_library();
	SDL_Surface* image = IMG_Load( filename.c_str() );
	if (!image)
	{
		NV_LOG( LOG_ERROR, "Image file " << filename.c_str() << " not found!" );
		return nullptr;
	}
	image_data* data = new image_data( glm::ivec2( image->w, image->h ), image->format->BytesPerPixel, (nv::uint8*)image->pixels );
	return data;
}

texture2d* gl_device::create_texture2d( ivec2 size, image_format aformat, datatype adatatype, sampler asampler, void* data /*= nullptr */ )
{
	return new gl_texture2d( size, aformat, adatatype, asampler, data );
}

uint32 gl_device::get_ticks()
{
	return SDL_GetTicks();
}

void gl_device::delay( uint32 ms )
{
	return SDL_Delay( ms );
}

gl_device::~gl_device()
{
	SDL_Quit();
}
