// Copyright (C) 2012-2013 ChaosForge / Kornel Kisielewicz
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

#ifndef NV_LIB_SDL_HH
#define NV_LIB_SDL_HH

#include <nv/common.hh>

#define NV_SDL_DYNAMIC
//#define NV_SDL_SHARED

#define NV_SDL_C      0
#define NV_SDL_12     1
#define NV_SDL_20     2

#if !defined(NV_SDL_VERSION)
#	define NV_SDL_VERSION     NV_SDL_12
#endif

#if NV_PLATFORM == NV_WINDOWS
#	define NV_SDL_PATH_12 "SDL.dll"
#	define NV_SDL_PATH_20 "SDL2.dll"
#elif NV_PLATFORM == NV_APPLE
#	define NV_SDL_PATH_12 "SDL.framework/SDL"
#	define NV_SDL_PATH_20 "SDL2.framework/SDL"
#else
#	define NV_SDL_PATH_12 "libSDL-1.2.so.0"
#	define NV_SDL_PATH_20 "libSDL-1.2.so.0"
#endif

#if NV_SDL_VERSION == NV_SDL_12
#	define NV_SDL_PATH NV_SDL_PATH_12
#elif NV_SDL_VERSION == NV_SDL_20 
#	define NV_SDL_PATH NV_SDL_PATH_20
#elif NV_SDL_VERSION == NV_SDL_C
#	define NV_SDL_PATH nullptr
#else
#	error "Unrecognized NV_SDL_VERSION!"
#endif

#include <stdint.h>
#include <stdio.h>

extern "C" {

#if NV_PLATFORM == NV_WINDOWS 
#	define NV_SDL_APIENTRY __cdecl
#else
#	define NV_SDL_APIENTRY 
#endif

#if defined(NV_SDL_SHARED) && (NV_PLATFORM == NV_WINDOWS)
#	define NV_SDL_API __declspec(dllimport)
#else
#	define NV_SDL_API extern
#endif

/* SDL_stdinc.h types */
typedef enum {
	SDL_FALSE = 0,
	SDL_TRUE  = 1
} SDL_bool;

typedef int8_t		Sint8;
typedef uint8_t		Uint8;
typedef int16_t		Sint16;
typedef uint16_t	Uint16;
typedef int32_t		Sint32;
typedef uint32_t	Uint32;
typedef int64_t		Sint64;
typedef uint64_t	Uint64;

static_assert(sizeof(Uint8) == 1, "Uint8 has wrong size!" );
static_assert(sizeof(Sint8) == 1, "Sint8 has wrong size!");
static_assert(sizeof(Uint16) == 2, "Uint16 has wrong size!");
static_assert(sizeof(Sint16) == 2, "Sint16 has wrong size!");
static_assert(sizeof(Uint32) == 4, "Uint32 has wrong size!");
static_assert(sizeof(Sint32) == 4, "Sint32 has wrong size!");
static_assert(sizeof(Uint64) == 8, "Uint64 has wrong size!");
static_assert(sizeof(Sint64) == 8, "Sint64 has wrong size!");

#define RW_SEEK_SET	0
#define RW_SEEK_CUR	1
#define RW_SEEK_END	2

/* SDL.h defines */

#define	SDL_INIT_TIMER		0x00000001
#define SDL_INIT_AUDIO		0x00000010
#define SDL_INIT_VIDEO		0x00000020
#define SDL_INIT_JOYSTICK	0x00000200
#if NV_SDL_VERSION == NV_SDL_20
#define SDL_INIT_NOPARACHUTE    0x00100000
#define SDL_INIT_EVENTTHREAD    0x01000000
#define SDL_INIT_EVERYTHING     0x0000FFFF 
#else
#define SDL_INIT_HAPTIC         0x00001000
#define SDL_INIT_GAMECONTROLLER 0x00002000 
#define SDL_INIT_EVENTS         0x00004000
#define SDL_INIT_NOPARACHUTE    0x00100000
#define SDL_INIT_EVERYTHING ( \
	SDL_INIT_TIMER | SDL_INIT_AUDIO | SDL_INIT_VIDEO | SDL_INIT_EVENTS | \
	SDL_INIT_JOYSTICK | SDL_INIT_HAPTIC | SDL_INIT_GAMECONTROLLER \
	) 
#endif
/* SDL_error.h defines */
typedef enum {
	SDL_ENOMEM,
	SDL_EFREAD,
	SDL_EFWRITE,
	SDL_EFSEEK,
	SDL_UNSUPPORTED,
	SDL_LASTERROR
} SDL_errorcode;

#if NV_SDL_VERSION == NV_SDL_20
#define SDL_RWOPS_UNKNOWN   0
#define SDL_RWOPS_WINFILE   1
#define SDL_RWOPS_STDFILE   2
#define SDL_RWOPS_JNIFILE   3
#define SDL_RWOPS_MEMORY    4
#define SDL_RWOPS_MEMORY_RO 5
#endif

/* SDL_rwops.h defines */
typedef struct SDL_RWops {
#if NV_SDL_VERSION == NV_SDL_20
	Sint64 (NV_SDL_APIENTRY *size)(struct SDL_RWops *context);
	Sint64 (NV_SDL_APIENTRY *seek)(struct SDL_RWops *context, Sint64 offset, int whence);
	size_t (NV_SDL_APIENTRY *read)(struct SDL_RWops *context, void *ptr, size_t size, size_t maxnum);
	size_t (NV_SDL_APIENTRY *write)(struct SDL_RWops *context, const void *ptr, size_t size, size_t num);
#else
	int (NV_SDL_APIENTRY *seek)(struct SDL_RWops *context, int offset, int whence);
	int (NV_SDL_APIENTRY *read)(struct SDL_RWops *context, void *ptr, int size, int maxnum);
	int (NV_SDL_APIENTRY *write)(struct SDL_RWops *context, const void *ptr, int size, int num);
#endif
	int (NV_SDL_APIENTRY *close)(struct SDL_RWops *context);

	Uint32 type;
	union {
#if NV_PLATFORM == NV_WINDOWS 
	    struct {
		int   append;
		void *h;
		struct {
		    void *data;
		    int size;
		    int left;
		} buffer;
	    } win32io;
#endif
	    struct {
		int autoclose;
	 	FILE *fp;
	    } stdio;
	    struct {
		Uint8 *base;
	 	Uint8 *here;
		Uint8 *stop;
	    } mem;
	    struct {
		void *data1;
		void *data2;
	    } unknown;
	} hidden;

} SDL_RWops;

/* SDL_video.h defines */
#define SDL_ALPHA_OPAQUE 255
#define SDL_ALPHA_TRANSPARENT 0

#if NV_SDL_VERSION == NV_SDL_20
typedef struct
{
	int x;
	int y;
} SDL_Point;

typedef struct SDL_Rect {
	int x, y;
	int w, h;
} SDL_Rect;
#else
typedef struct SDL_Rect {
	Sint16 x, y;
	Uint16 w, h;
} SDL_Rect;
#endif

typedef struct SDL_Color {
	Uint8 r;
	Uint8 g;
	Uint8 b;
#if NV_SDL_VERSION == NV_SDL_20
	Uint8 unused;
#else
	Uint8 a;
#endif
} SDL_Color;

typedef struct SDL_Palette {
	int       ncolors;
	SDL_Color *colors;
#if NV_SDL_VERSION == NV_SDL_20
	Uint32    version;
	int       refcount;
#endif

} SDL_Palette;

#if NV_SDL_VERSION == NV_SDL_20
typedef struct SDL_PixelFormat {
	Uint32 format;
	SDL_Palette *palette;
	Uint8  BitsPerPixel;
	Uint8  BytesPerPixel;
	Uint8  padding[2];
	Uint32 Rmask;
	Uint32 Gmask;
	Uint32 Bmask;
	Uint32 Amask;
	Uint8 Rloss;
	Uint8 Gloss;
	Uint8 Bloss;
	Uint8 Aloss;
	Uint8 Rshift;
	Uint8 Gshift;
	Uint8 Bshift;
	Uint8 Ashift;
	int refcount;
	struct SDL_PixelFormat *next; 
} SDL_PixelFormat;
#else
typedef struct SDL_PixelFormat {
	SDL_Palette *palette;
	Uint8  BitsPerPixel;
	Uint8  BytesPerPixel;
	Uint8  Rloss;
	Uint8  Gloss;
	Uint8  Bloss;
	Uint8  Aloss;
	Uint8  Rshift;
	Uint8  Gshift;
	Uint8  Bshift;
	Uint8  Ashift;
	Uint32 Rmask;
	Uint32 Gmask;
	Uint32 Bmask;
	Uint32 Amask;

	Uint32 colorkey;
	Uint8  alpha;
} SDL_PixelFormat;
#endif

#if NV_SDL_VERSION == NV_SDL_20
typedef struct
{
	Uint32 format; 
	int w;
	int h;
	int refresh_rate;
	void *driverdata;
} SDL_DisplayMode;
#endif

#if NV_SDL_VERSION == NV_SDL_20
typedef struct SDL_Surface 
{
	Uint32 flags;
	SDL_PixelFormat *format;
	int w, h;
	int pitch;
	void *pixels;
	void *userdata;
	int locked;
	void *lock_data;
	SDL_Rect clip_rect;
	struct SDL_BlitMap *map;
	int refcount;
} SDL_Surface;
#else
typedef struct SDL_Surface
{
	Uint32 flags;
	SDL_PixelFormat *format;
	int w, h;
	Uint16 pitch;
	void *pixels;
	int offset;	
	struct private_hwdata *hwdata;
	SDL_Rect clip_rect;
	Uint32 unused1;
	Uint32 locked;
	struct SDL_BlitMap *map;
	unsigned int format_version;
	int refcount;
} SDL_Surface;
#endif

#if NV_SDL_VERSION == NV_SDL_20
#define SDL_SWSURFACE       0         
#define SDL_PREALLOC        0x00000001
#define SDL_RLEACCEL        0x00000002
#define SDL_DONTFREE        0x00000004

#define SDL_MUSTLOCK(S) (((S)->flags & SDL_RLEACCEL) != 0)
#else
#define SDL_SWSURFACE	0x00000000	
#define SDL_HWSURFACE	0x00000001	
#define SDL_ASYNCBLIT	0x00000004	
#define SDL_ANYFORMAT	0x10000000	
#define SDL_HWPALETTE	0x20000000	
#define SDL_DOUBLEBUF	0x40000000	
#define SDL_FULLSCREEN	0x80000000	
#define SDL_OPENGL      0x00000002  
#define SDL_OPENGLBLIT	0x0000000A	
#define SDL_RESIZABLE	0x00000010	
#define SDL_NOFRAME	0x00000020	
#define SDL_HWACCEL	0x00000100	
#define SDL_SRCCOLORKEY	0x00001000	
#define SDL_RLEACCELOK	0x00002000	
#define SDL_RLEACCEL	0x00004000	
#define SDL_SRCALPHA	0x00010000	
#define SDL_PREALLOC	0x01000000	

#define SDL_MUSTLOCK(surface)	\
	(surface->offset ||		\
	((surface->flags & (SDL_HWSURFACE|SDL_ASYNCBLIT|SDL_RLEACCEL)) != 0))

#endif

typedef int (*SDL_blit)(struct SDL_Surface *src, SDL_Rect *srcrect, 
                        struct SDL_Surface *dst, SDL_Rect *dstrect);

#if NV_SDL_VERSION == NV_SDL_12
typedef struct SDL_VideoInfo {
	Uint32 hw_available :1;
	Uint32 wm_available :1;
	Uint32 UnusedBits1  :6;
	Uint32 UnusedBits2  :1;
	Uint32 blit_hw      :1;
	Uint32 blit_hw_CC   :1;
	Uint32 blit_hw_A    :1;
	Uint32 blit_sw      :1;
	Uint32 blit_sw_CC   :1;
	Uint32 blit_sw_A    :1;
	Uint32 blit_fill    :1;
	Uint32 UnusedBits3  :16;
	Uint32 video_mem;
	SDL_PixelFormat *vfmt;
	int    current_w;
	int    current_h;
} SDL_VideoInfo;
#endif

#if NV_SDL_VERSION == NV_SDL_20
typedef enum
{
    SDL_WINDOW_FULLSCREEN = 0x00000001,
    SDL_WINDOW_OPENGL = 0x00000002,
    SDL_WINDOW_SHOWN = 0x00000004,
    SDL_WINDOW_HIDDEN = 0x00000008,
    SDL_WINDOW_BORDERLESS = 0x00000010,
    SDL_WINDOW_RESIZABLE = 0x00000020,
    SDL_WINDOW_MINIMIZED = 0x00000040,
    SDL_WINDOW_MAXIMIZED = 0x00000080,
    SDL_WINDOW_INPUT_GRABBED = 0x00000100,
    SDL_WINDOW_INPUT_FOCUS = 0x00000200,
    SDL_WINDOW_MOUSE_FOCUS = 0x00000400,
    SDL_WINDOW_FULLSCREEN_DESKTOP = ( SDL_WINDOW_FULLSCREEN | 0x00001000 ),
    SDL_WINDOW_FOREIGN = 0x00000800 
} SDL_WindowFlags;

#define SDL_WINDOWPOS_UNDEFINED_MASK    0x1FFF0000
#define SDL_WINDOWPOS_UNDEFINED_DISPLAY(X)  (SDL_WINDOWPOS_UNDEFINED_MASK|(X))
#define SDL_WINDOWPOS_UNDEFINED         SDL_WINDOWPOS_UNDEFINED_DISPLAY(0)
#define SDL_WINDOWPOS_ISUNDEFINED(X)    \
            (((X)&0xFFFF0000) == SDL_WINDOWPOS_UNDEFINED_MASK)

#define SDL_WINDOWPOS_CENTERED_MASK    0x2FFF0000
#define SDL_WINDOWPOS_CENTERED_DISPLAY(X)  (SDL_WINDOWPOS_CENTERED_MASK|(X))
#define SDL_WINDOWPOS_CENTERED         SDL_WINDOWPOS_CENTERED_DISPLAY(0)
#define SDL_WINDOWPOS_ISCENTERED(X)    \
            (((X)&0xFFFF0000) == SDL_WINDOWPOS_CENTERED_MASK)

typedef enum
{
    SDL_WINDOWEVENT_NONE,
    SDL_WINDOWEVENT_SHOWN,
    SDL_WINDOWEVENT_HIDDEN,
    SDL_WINDOWEVENT_EXPOSED,
    SDL_WINDOWEVENT_MOVED,
    SDL_WINDOWEVENT_RESIZED,
    SDL_WINDOWEVENT_SIZE_CHANGED,
    SDL_WINDOWEVENT_MINIMIZED,
    SDL_WINDOWEVENT_MAXIMIZED,
    SDL_WINDOWEVENT_RESTORED,
    SDL_WINDOWEVENT_ENTER,
    SDL_WINDOWEVENT_LEAVE,
    SDL_WINDOWEVENT_FOCUS_GAINED,
    SDL_WINDOWEVENT_FOCUS_LOST,
    SDL_WINDOWEVENT_CLOSE
} SDL_WindowEventID;

typedef void *SDL_GLContext;
#endif


typedef enum {
    SDL_GL_RED_SIZE,
    SDL_GL_GREEN_SIZE,
    SDL_GL_BLUE_SIZE,
    SDL_GL_ALPHA_SIZE,
    SDL_GL_BUFFER_SIZE,
    SDL_GL_DOUBLEBUFFER,
    SDL_GL_DEPTH_SIZE,
    SDL_GL_STENCIL_SIZE,
    SDL_GL_ACCUM_RED_SIZE,
    SDL_GL_ACCUM_GREEN_SIZE,
    SDL_GL_ACCUM_BLUE_SIZE,
    SDL_GL_ACCUM_ALPHA_SIZE,
    SDL_GL_STEREO,
    SDL_GL_MULTISAMPLEBUFFERS,
    SDL_GL_MULTISAMPLESAMPLES,
    SDL_GL_ACCELERATED_VISUAL,
#if NV_SDL_VERSION == NV_SDL_12
    SDL_GL_SWAP_CONTROL
#else
	SDL_GL_RETAINED_BACKING,
	SDL_GL_CONTEXT_MAJOR_VERSION,
	SDL_GL_CONTEXT_MINOR_VERSION,
	SDL_GL_CONTEXT_EGL,
	SDL_GL_CONTEXT_FLAGS,
	SDL_GL_CONTEXT_PROFILE_MASK,
	SDL_GL_SHARE_WITH_CURRENT_CONTEXT
#endif
} SDL_GLattr;

#if NV_SDL_VERSION == NV_SDL_20
typedef enum
{
	SDL_GL_CONTEXT_PROFILE_CORE           = 0x0001,
	SDL_GL_CONTEXT_PROFILE_COMPATIBILITY  = 0x0002,
	SDL_GL_CONTEXT_PROFILE_ES             = 0x0004
} SDL_GLprofile;

typedef enum
{
	SDL_GL_CONTEXT_DEBUG_FLAG              = 0x0001,
	SDL_GL_CONTEXT_FORWARD_COMPATIBLE_FLAG = 0x0002,
	SDL_GL_CONTEXT_ROBUST_ACCESS_FLAG      = 0x0004,
	SDL_GL_CONTEXT_RESET_ISOLATION_FLAG    = 0x0008
} SDL_GLcontextFlag;
#endif

#if NV_SDL_VERSION == NV_SDL_12
typedef enum {
	SDL_GRAB_QUERY = -1,
	SDL_GRAB_OFF = 0,
	SDL_GRAB_ON = 1,
	SDL_GRAB_FULLSCREEN	/**< Used internally */
} SDL_GrabMode;
#endif

/* SDL_audio.h defines */
#define AUDIO_U8	0x0008
#define AUDIO_S8	0x8008
#define AUDIO_U16LSB	0x0010	
#define AUDIO_S16LSB	0x8010	
#define AUDIO_U16MSB	0x1010	
#define AUDIO_S16MSB	0x9010
#define AUDIO_U16	AUDIO_U16LSB
#define AUDIO_S16	AUDIO_S16LSB

#if NV_SDL_VERSION == NV_SDL_20
#define AUDIO_S32LSB    0x8020 
#define AUDIO_S32MSB    0x9020 
#define AUDIO_S32       AUDIO_S32LSB
#define AUDIO_F32LSB    0x8120 
#define AUDIO_F32MSB    0x9120 
#define AUDIO_F32       AUDIO_F32LSB
#endif

/* NOTE : assuming LSB! */
#define AUDIO_U16SYS	AUDIO_U16LSB
#define AUDIO_S16SYS	AUDIO_S16LSB
#if NV_SDL_VERSION == NV_SDL_20
#define AUDIO_S32SYS    AUDIO_S32LSB
#define AUDIO_F32SYS    AUDIO_F32LSB
#endif

typedef Uint16 SDL_AudioFormat;
typedef void (NV_SDL_APIENTRY * SDL_AudioCallback) (void *userdata, Uint8 * stream, int len);
struct SDL_AudioCVT;
typedef void (NV_SDL_APIENTRY * SDL_AudioFilter) (struct SDL_AudioCVT * cvt, SDL_AudioFormat format);

typedef struct SDL_AudioSpec
{
	int freq;
	SDL_AudioFormat format;
	Uint8 channels;
	Uint8 silence;
	Uint16 samples;
	Uint16 padding;
	Uint32 size;
	SDL_AudioCallback callback;
	void *userdata;
} SDL_AudioSpec;

typedef struct SDL_AudioCVT {
	int needed;
	SDL_AudioFormat src_format;
	SDL_AudioFormat dst_format;
	double rate_incr;
	Uint8 *buf;
	int len;
	int len_cvt;
	int len_mult;
	double len_ratio;
	SDL_AudioFilter filters[10];
	int filter_index;
} SDL_AudioCVT; 

typedef enum {
	SDL_AUDIO_STOPPED = 0,
	SDL_AUDIO_PLAYING,
	SDL_AUDIO_PAUSED
} SDL_audiostatus;

typedef SDL_audiostatus SDL_AudioStatus;

#if NV_SDL_VERSION == NV_SDL_20
#define SDL_AUDIO_ALLOW_FREQUENCY_CHANGE    0x00000001
#define SDL_AUDIO_ALLOW_FORMAT_CHANGE       0x00000002
#define SDL_AUDIO_ALLOW_CHANNELS_CHANGE     0x00000004
#define SDL_AUDIO_ALLOW_ANY_CHANGE          (SDL_AUDIO_ALLOW_FREQUENCY_CHANGE|SDL_AUDIO_ALLOW_FORMAT_CHANGE|SDL_AUDIO_ALLOW_CHANNELS_CHANGE)
#endif

#define SDL_MIX_MAXVOLUME 128

/* SDL_active.h defines */
#if NV_SDL_VERSION == NV_SDL_12
#define SDL_APPMOUSEFOCUS	0x01
#define SDL_APPINPUTFOCUS	0x02
#define SDL_APPACTIVE		0x04
#endif

/* SDL_keyboard.h defines */
#if NV_SDL_VERSION == NV_SDL_20
#include <nv/lib/detail/sdl_keys_20.inc>
#else
#include <nv/lib/detail/sdl_keys_12.inc>
#endif

typedef enum {
	KMOD_NONE  = 0x0000,
	KMOD_LSHIFT= 0x0001,
	KMOD_RSHIFT= 0x0002,
	KMOD_LCTRL = 0x0040,
	KMOD_RCTRL = 0x0080,
	KMOD_LALT  = 0x0100,
	KMOD_RALT  = 0x0200,
	KMOD_LMETA = 0x0400,
	KMOD_RMETA = 0x0800,
	KMOD_LGUI  = 0x0400,
	KMOD_RGUI  = 0x0800,
	KMOD_NUM   = 0x1000,
	KMOD_CAPS  = 0x2000,
	KMOD_MODE  = 0x4000,
	KMOD_RESERVED = 0x8000
} SDLMod;

#define KMOD_CTRL	(KMOD_LCTRL|KMOD_RCTRL)
#define KMOD_SHIFT	(KMOD_LSHIFT|KMOD_RSHIFT)
#define KMOD_ALT	(KMOD_LALT|KMOD_RALT)
#define KMOD_META	(KMOD_LMETA|KMOD_RMETA)
#define KMOD_GUI	(KMOD_LGUI|KMOD_RGUI)

#define SDL_DEFAULT_REPEAT_DELAY	500
#define SDL_DEFAULT_REPEAT_INTERVAL	30

#if NV_SDL_VERSION == NV_SDL_12
typedef struct SDL_keysym {
	Uint8 scancode;
	SDLKey sym;
	SDLMod mod;
	Uint16 unicode;
} SDL_keysym;

#define SDL_ALL_HOTKEYS		0xFFFFFFFF
#else
typedef struct SDL_Keysym
{
	SDL_Scancode scancode;
	SDL_Keycode sym;
	Uint16 mod;
	Uint32 unused;
} SDL_Keysym;
#endif

/* SDL_mouse.h defines */
#if NV_SDL_VERSION == NV_SDL_20
typedef enum
{
	SDL_SYSTEM_CURSOR_ARROW,
	SDL_SYSTEM_CURSOR_IBEAM,
	SDL_SYSTEM_CURSOR_WAIT,
	SDL_SYSTEM_CURSOR_CROSSHAIR,
	SDL_SYSTEM_CURSOR_WAITARROW,
	SDL_SYSTEM_CURSOR_SIZENWSE,
	SDL_SYSTEM_CURSOR_SIZENESW,
	SDL_SYSTEM_CURSOR_SIZEWE,
	SDL_SYSTEM_CURSOR_SIZENS,
	SDL_SYSTEM_CURSOR_SIZEALL,
	SDL_SYSTEM_CURSOR_NO,
	SDL_SYSTEM_CURSOR_HAND,
	SDL_NUM_SYSTEM_CURSORS
} SDL_SystemCursor;
#endif

typedef struct WMcursor WMcursor;
typedef struct SDL_Cursor {
	SDL_Rect area;	
	Sint16 hot_x, hot_y;
	Uint8 *data;
	Uint8 *mask;
	Uint8 *save[2];	
	WMcursor *wm_cursor;
} SDL_Cursor;
 
#define SDL_BUTTON(X)		(1 << ((X)-1))
#define SDL_BUTTON_LEFT		1
#define SDL_BUTTON_MIDDLE	2
#define SDL_BUTTON_RIGHT	3
#if NV_SDL_VERSION == NV_SDL_20
#define SDL_BUTTON_X1		4
#define SDL_BUTTON_X2		5
#else
#define SDL_BUTTON_WHEELUP	4
#define SDL_BUTTON_WHEELDOWN	5
#define SDL_BUTTON_X1		6
#define SDL_BUTTON_X2		7
#endif
#define SDL_BUTTON_LMASK	SDL_BUTTON(SDL_BUTTON_LEFT)
#define SDL_BUTTON_MMASK	SDL_BUTTON(SDL_BUTTON_MIDDLE)
#define SDL_BUTTON_RMASK	SDL_BUTTON(SDL_BUTTON_RIGHT)
#define SDL_BUTTON_X1MASK	SDL_BUTTON(SDL_BUTTON_X1)
#define SDL_BUTTON_X2MASK	SDL_BUTTON(SDL_BUTTON_X2) 

/* SDL_joystick.h defines */

struct _SDL_Joystick;
typedef struct _SDL_Joystick SDL_Joystick; 

#if NV_SDL_VERSION == NV_SDL_20
typedef struct {
	Uint8 data[16];
} SDL_JoystickGUID;

typedef Sint32 SDL_JoystickID;
#endif


#define SDL_HAT_CENTERED	0x00
#define SDL_HAT_UP		0x01
#define SDL_HAT_RIGHT		0x02
#define SDL_HAT_DOWN		0x04
#define SDL_HAT_LEFT		0x08
#define SDL_HAT_RIGHTUP		(SDL_HAT_RIGHT|SDL_HAT_UP)
#define SDL_HAT_RIGHTDOWN	(SDL_HAT_RIGHT|SDL_HAT_DOWN)
#define SDL_HAT_LEFTUP		(SDL_HAT_LEFT|SDL_HAT_UP)
#define SDL_HAT_LEFTDOWN	(SDL_HAT_LEFT|SDL_HAT_DOWN)

/* SDL_event.h defines */

#define SDL_RELEASED	0
#define SDL_PRESSED	1

#if NV_SDL_VERSION == NV_SDL_12
#include <nv/lib/detail/sdl_events_12.inc>
#else
#include <nv/lib/detail/sdl_events_20.inc>
#endif


#define SDL_QUERY	-1
#define SDL_IGNORE	 0
#define SDL_DISABLE	 0
#define SDL_ENABLE	 1

/* SDL_timer.h defines */
#if  NV_SDL_VERSION == NV_SDL_20
typedef Uint32 (NV_SDL_APIENTRY *SDL_TimerCallback)(Uint32 interval, void *param);
typedef int SDL_TimerID;
#else
#define SDL_TIMESLICE		10
#define TIMER_RESOLUTION	10
typedef Uint32 (NV_SDL_APIENTRY *SDL_TimerCallback)(Uint32 interval);
typedef Uint32 (NV_SDL_APIENTRY *SDL_NewTimerCallback)(Uint32 interval, void *param);
typedef struct _SDL_TimerID *SDL_TimerID;
#endif

/* SDL_version.h defines */
typedef struct SDL_version {
	Uint8 major;
	Uint8 minor;
	Uint8 patch;
} SDL_version;

#if  NV_SDL_VERSION == NV_SDL_20
#define SDL_MAJOR_VERSION   2
#define SDL_MINOR_VERSION   0
#define SDL_PATCHLEVEL      0 
#else
#define SDL_MAJOR_VERSION   1
#define SDL_MINOR_VERSION   2
#define SDL_PATCHLEVEL      15 
#endif

#if defined(NV_SDL_DYNAMIC)
#	define NV_SDL_FUN( rtype, fname, fparams ) NV_SDL_API rtype (NV_SDL_APIENTRY *fname) fparams
#else
#	define NV_SDL_FUN( rtype, fname, fparams ) NV_SDL_API rtype NV_SDL_APIENTRY fname fparams
#endif
#include <nv/lib/detail/sdl_functions.inc>
#undef NV_SDL_FUN

#define SDL_RWseek(ctx, offset, whence)	(ctx)->seek(ctx, offset, whence)
#define SDL_RWtell(ctx)			(ctx)->seek(ctx, 0, RW_SEEK_CUR)
#define SDL_RWread(ctx, ptr, size, n)	(ctx)->read(ctx, ptr, size, n)
#define SDL_RWwrite(ctx, ptr, size, n)	(ctx)->write(ctx, ptr, size, n)
#define SDL_RWclose(ctx)		(ctx)->close(ctx)
#define SDL_BlitSurface SDL_UpperBlit
#define SDL_VERSION(X)							\
{									\
	(X)->major = SDL_MAJOR_VERSION;					\
	(X)->minor = SDL_MINOR_VERSION;					\
	(X)->patch = SDL_PATCHLEVEL;					\
}
#define SDL_VERSIONNUM(X, Y, Z)						\
	((X)*1000 + (Y)*100 + (Z))
#define SDL_COMPILEDVERSION \
	SDL_VERSIONNUM(SDL_MAJOR_VERSION, SDL_MINOR_VERSION, SDL_PATCHLEVEL)
#define SDL_VERSION_ATLEAST(X, Y, Z) \
	(SDL_COMPILEDVERSION >= SDL_VERSIONNUM(X, Y, Z))
 
}
#define SDL_VERSIONNUM(X, Y, Z)                     \
	((X)*1000 + (Y)*100 + (Z)) 
#define SDL_COMPILEDVERSION \
	SDL_VERSIONNUM(SDL_MAJOR_VERSION, SDL_MINOR_VERSION, SDL_PATCHLEVEL)
#define SDL_VERSION_ATLEAST(X, Y, Z) \
	(SDL_COMPILEDVERSION >= SDL_VERSIONNUM(X, Y, Z))

namespace nv {
/* Dynamic load support */
#	if defined( NV_SDL_DYNAMIC )
		bool load_sdl_library( const char* path = NV_SDL_PATH );
#	else
		inline bool load_sdl_library( const char* path = "" ) { return true; }
#	endif
}

#endif // NV_LIB_SDL_HH
