// Copyright (C) 2012-2013 ChaosForge / Kornel Kisielewicz
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

#ifndef NV_LIB_LUA_HH
#define NV_LIB_LUA_HH

// Portions of the official Lua 5.2 headers were used, copyright follows:

/******************************************************************************
* Copyright (C) 1994-2012 Lua.org, PUC-Rio.
*
* Permission is hereby granted, free of charge, to any person obtaining
* a copy of this software and associated documentation files (the
* "Software"), to deal in the Software without restriction, including
* without limitation the rights to use, copy, modify, merge, publish,
* distribute, sublicense, and/or sell copies of the Software, and to
* permit persons to whom the Software is furnished to do so, subject to
* the following conditions:
*
* The above copyright notice and this permission notice shall be
* included in all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
* EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
* IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
* CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
* TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
* SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
******************************************************************************/

#include <nv/common.hh>

#define NV_LUA_DYNAMIC
//#define NV_LUA_SHARED

#if NV_PLATFORM == NV_WINDOWS
#	define NV_LUA_PATH "lua52.dll"
#elif NV_PLATFORM == NV_APPLE
#	define NV_LUA_PATH "lua5.2.dylib"
#else
#	define NV_LUA_PATH "lua5.2.so"
#endif

#include <limits.h>
#include <stdarg.h>
#include <stddef.h>
#include <stdio.h>

extern "C" {

#if defined(NV_LUA_SHARED) && (NV_PLATFORM == NV_WINDOWS)
#	define NV_LUA_API __declspec(dllimport)
#else
#	define NV_LUA_API extern
#endif

#if defined(NV_LUA_DYNAMIC)
#	define NV_LUA_FUN( rtype, fname, fparams ) NV_LUA_API rtype (*fname) fparams
#else
#	define NV_LUA_FUN( rtype, fname, fparams ) NV_LUA_API rtype fname fparams
#endif

/* luaconf.h definitions (only the ones needed in the headers) */
#define LUA_NUMBER   double
#define LUA_INTEGER  ptrdiff_t
#define LUA_UNSIGNED unsigned int

#define LUAI_MAXSTACK		1000000
#define LUAI_FIRSTPSEUDOIDX	(-LUAI_MAXSTACK - 1000)
#define LUA_IDSIZE 60

/* lua.h */
#define LUA_VERSION_MAJOR	"5"
#define LUA_VERSION_MINOR	"2"
#define LUA_VERSION_NUM		502
#define LUA_VERSION_RELEASE	"1"

#define LUA_VERSION	"Lua " LUA_VERSION_MAJOR "." LUA_VERSION_MINOR
#define LUA_RELEASE	LUA_VERSION "." LUA_VERSION_RELEASE
#define LUA_COPYRIGHT	LUA_RELEASE "  Copyright (C) 1994-2012 Lua.org, PUC-Rio"
#define LUA_AUTHORS	"R. Ierusalimschy, L. H. de Figueiredo, W. Celes"

#define LUA_SIGNATURE	"\033Lua"
#define LUA_MULTRET	(-1)
#define LUA_NOREF       (-2)
#define LUA_REFNIL      (-1)

#define LUA_REGISTRYINDEX	LUAI_FIRSTPSEUDOIDX
#define lua_upvalueindex(i)	(LUA_REGISTRYINDEX - (i))

#define LUAL_BUFFERSIZE		BUFSIZ


/* Types */
typedef struct lua_State lua_State;
typedef struct lua_Debug lua_Debug; 
typedef int (*lua_CFunction) (lua_State *L);
typedef const char * (*lua_Reader) (lua_State *L, void *ud, size_t *sz);
typedef int (*lua_Writer) (lua_State *L, const void* p, size_t sz, void* ud);
typedef void * (*lua_Alloc) (void *ud, void *ptr, size_t osize, size_t nsize);
typedef void (*lua_Hook) (lua_State *L, lua_Debug *ar);

struct lua_Debug {
  int event;
  const char *name;	/* (n) */
  const char *namewhat;	/* (n) 'global', 'local', 'field', 'method' */
  const char *what;	/* (S) 'Lua', 'C', 'main', 'tail' */
  const char *source;	/* (S) */
  int currentline;	/* (l) */
  int linedefined;	/* (S) */
  int lastlinedefined;	/* (S) */
  unsigned char nups;	/* (u) number of upvalues */
  unsigned char nparams;/* (u) number of parameters */
  char isvararg;        /* (u) */
  char istailcall;	/* (t) */
  char short_src[LUA_IDSIZE]; /* (S) */
  struct CallInfo *i_ci;  /* active function */
};

/* Aux lib structs */
typedef struct luaL_Reg {
  const char *name;
  lua_CFunction func;
} luaL_Reg;

typedef struct luaL_Buffer {
  char *b;  /* buffer address */
  size_t size;  /* buffer size */
  size_t n;  /* number of characters in buffer */
  lua_State *L;
  char initb[LUAL_BUFFERSIZE];  /* initial buffer */
} luaL_Buffer;

typedef struct luaL_Stream {
  FILE *f;  /* stream (NULL for incompletely created streams) */
  lua_CFunction closef;  /* to close stream (NULL for closed streams) */
} luaL_Stream;

typedef LUA_NUMBER lua_Number;
typedef LUA_INTEGER lua_Integer;
typedef LUA_UNSIGNED lua_Unsigned;

/* Thread return codes */
#define LUA_OK		0
#define LUA_YIELD	1
#define LUA_ERRRUN	2
#define LUA_ERRSYNTAX	3
#define LUA_ERRMEM	4
#define LUA_ERRGCMM	5
#define LUA_ERRERR	6
#define LUA_ERRFILE 7


/* Type codes */
#define LUA_TNONE			(-1)
#define LUA_TNIL			0
#define LUA_TBOOLEAN		1
#define LUA_TLIGHTUSERDATA	2
#define LUA_TNUMBER			3
#define LUA_TSTRING			4
#define LUA_TTABLE			5
#define LUA_TFUNCTION		6
#define LUA_TUSERDATA		7
#define LUA_TTHREAD			8

#define LUA_NUMTAGS			9
#define LUA_MINSTACK		20

/* Pre-defined registers */
#define LUA_RIDX_MAINTHREAD	1
#define LUA_RIDX_GLOBALS	2
#define LUA_RIDX_LAST		LUA_RIDX_GLOBALS

/* OP codes */
#define LUA_OPADD	0
#define LUA_OPSUB	1
#define LUA_OPMUL	2
#define LUA_OPDIV	3
#define LUA_OPMOD	4
#define LUA_OPPOW	5
#define LUA_OPUNM	6

/* Compare codes */
#define LUA_OPEQ	0
#define LUA_OPLT	1
#define LUA_OPLE	2

/* Event codes */
#define LUA_GCSTOP			0
#define LUA_GCRESTART		1
#define LUA_GCCOLLECT		2
#define LUA_GCCOUNT			3
#define LUA_GCCOUNTB		4
#define LUA_GCSTEP			5
#define LUA_GCSETPAUSE		6
#define LUA_GCSETSTEPMUL	7
#define LUA_GCSETMAJORINC	8
#define LUA_GCISRUNNING		9
#define LUA_GCGEN			10
#define LUA_GCINC			11

/* Debug event codes */
#define LUA_HOOKCALL	0
#define LUA_HOOKRET	1
#define LUA_HOOKLINE	2
#define LUA_HOOKCOUNT	3
#define LUA_HOOKTAILCALL 4

/* Debug event masks */
#define LUA_MASKCALL	(1 << LUA_HOOKCALL)
#define LUA_MASKRET	(1 << LUA_HOOKRET)
#define LUA_MASKLINE	(1 << LUA_HOOKLINE)
#define LUA_MASKCOUNT	(1 << LUA_HOOKCOUNT)

/* lualib.h defines */
#define LUA_COLIBNAME	"coroutine"
#define LUA_TABLIBNAME	"table"
#define LUA_IOLIBNAME	"io"
#define LUA_OSLIBNAME	"os"
#define LUA_STRLIBNAME	"string"
#define LUA_BITLIBNAME	"bit32"
#define LUA_MATHLIBNAME	"math"
#define LUA_DBLIBNAME	"debug"
#define LUA_LOADLIBNAME	"package"

/* State manipulation */
NV_LUA_FUN( lua_State *,(lua_newstate),(lua_Alloc f, void *ud) );
NV_LUA_FUN( void       ,(lua_close),(lua_State *L) );
NV_LUA_FUN( lua_State *,(lua_newthread),(lua_State *L) );
NV_LUA_FUN( lua_CFunction,(lua_atpanic),(lua_State *L, lua_CFunction panicf) );
NV_LUA_FUN( const lua_Number *,(lua_version),(lua_State *L) );

/* Basic stack manipulation */
NV_LUA_FUN( int  ,(lua_absindex),(lua_State *L, int idx) );
NV_LUA_FUN( int  ,(lua_gettop),(lua_State *L) );
NV_LUA_FUN( void ,(lua_settop),(lua_State *L, int idx) );
NV_LUA_FUN( void ,(lua_pushvalue),(lua_State *L, int idx) );
NV_LUA_FUN( void ,(lua_remove),(lua_State *L, int idx) );
NV_LUA_FUN( void ,(lua_insert),(lua_State *L, int idx) );
NV_LUA_FUN( void ,(lua_replace),(lua_State *L, int idx) );
NV_LUA_FUN( void ,(lua_copy),(lua_State *L, int fromidx, int toidx) );
NV_LUA_FUN( int  ,(lua_checkstack),(lua_State *L, int sz) );
NV_LUA_FUN( void ,(lua_xmove),(lua_State *from, lua_State *to, int n) );

/* Access functions (stack -> C) */
NV_LUA_FUN( int            ,(lua_isnumber),(lua_State *L, int idx) );
NV_LUA_FUN( int            ,(lua_isstring),(lua_State *L, int idx) );
NV_LUA_FUN( int            ,(lua_iscfunction),(lua_State *L, int idx) );
NV_LUA_FUN( int            ,(lua_isuserdata),(lua_State *L, int idx) );
NV_LUA_FUN( int            ,(lua_type),(lua_State *L, int idx) );
NV_LUA_FUN( const char*    ,(lua_typename),(lua_State *L, int tp) );

NV_LUA_FUN( lua_Number     ,(lua_tonumberx),(lua_State *L, int idx, int *isnum) );
NV_LUA_FUN( lua_Integer    ,(lua_tointegerx),(lua_State *L, int idx, int *isnum) );
NV_LUA_FUN( lua_Unsigned   ,(lua_tounsignedx),(lua_State *L, int idx, int *isnum) );
NV_LUA_FUN( int            ,(lua_toboolean),(lua_State *L, int idx) );
NV_LUA_FUN( const char*    ,(lua_tolstring),(lua_State *L, int idx, size_t *len) );
NV_LUA_FUN( size_t         ,(lua_rawlen),(lua_State *L, int idx) );
NV_LUA_FUN( lua_CFunction  ,(lua_tocfunction),(lua_State *L, int idx) );
NV_LUA_FUN( void*          ,(lua_touserdata),(lua_State *L, int idx) );
NV_LUA_FUN( lua_State*     ,(lua_tothread),(lua_State *L, int idx) );
NV_LUA_FUN( const void*    ,(lua_topointer),(lua_State *L, int idx) );

/* Comparison and arithmetic functions */
NV_LUA_FUN( void ,(lua_arith),(lua_State *L, int op) );
NV_LUA_FUN( int  ,(lua_rawequal),(lua_State *L, int idx1, int idx2) );
NV_LUA_FUN( int  ,(lua_compare),(lua_State *L, int idx1, int idx2, int op) );

/* Push functions (C -> stack) */
NV_LUA_FUN( void        ,(lua_pushnil),(lua_State *L) );
NV_LUA_FUN( void        ,(lua_pushnumber),(lua_State *L, lua_Number n) );
NV_LUA_FUN( void        ,(lua_pushinteger),(lua_State *L, lua_Integer n) );
NV_LUA_FUN( void        ,(lua_pushunsigned),(lua_State *L, lua_Unsigned n) );
NV_LUA_FUN( const char* ,(lua_pushlstring),(lua_State *L, const char *s, size_t l) );
NV_LUA_FUN( const char* ,(lua_pushstring),(lua_State *L, const char *s) );
NV_LUA_FUN( const char* ,(lua_pushvfstring),(lua_State *L, const char *fmt,
                                                      va_list argp) );
NV_LUA_FUN( const char *,(lua_pushfstring),(lua_State *L, const char *fmt, ...) );
NV_LUA_FUN( void ,(lua_pushcclosure),(lua_State *L, lua_CFunction fn, int n) );
NV_LUA_FUN( void ,(lua_pushboolean),(lua_State *L, int b) );
NV_LUA_FUN( void ,(lua_pushlightuserdata),(lua_State *L, void *p) );
NV_LUA_FUN( int  ,(lua_pushthread),(lua_State *L) );

/* Get functions (Lua -> stack) */
NV_LUA_FUN( void ,(lua_getglobal),(lua_State *L, const char *var) );
NV_LUA_FUN( void ,(lua_gettable),(lua_State *L, int idx) );
NV_LUA_FUN( void ,(lua_getfield),(lua_State *L, int idx, const char *k) );
NV_LUA_FUN( void ,(lua_rawget),(lua_State *L, int idx) );
NV_LUA_FUN( void ,(lua_rawgeti),(lua_State *L, int idx, int n) );
NV_LUA_FUN( void ,(lua_rawgetp),(lua_State *L, int idx, const void *p) );
NV_LUA_FUN( void ,(lua_createtable),(lua_State *L, int narr, int nrec) );
NV_LUA_FUN( void*,(lua_newuserdata),(lua_State *L, size_t sz) );
NV_LUA_FUN( int  ,(lua_getmetatable),(lua_State *L, int objindex) );
NV_LUA_FUN( void ,(lua_getuservalue),(lua_State *L, int idx) );

/* Set functions (stack -> Lua) */
NV_LUA_FUN( void ,(lua_setglobal),(lua_State *L, const char *var) );
NV_LUA_FUN( void ,(lua_settable),(lua_State *L, int idx) );
NV_LUA_FUN( void ,(lua_setfield),(lua_State *L, int idx, const char *k) );
NV_LUA_FUN( void ,(lua_rawset),(lua_State *L, int idx) );
NV_LUA_FUN( void ,(lua_rawseti),(lua_State *L, int idx, int n) );
NV_LUA_FUN( void ,(lua_rawsetp),(lua_State *L, int idx, const void *p) );
NV_LUA_FUN( int  ,(lua_setmetatable),(lua_State *L, int objindex) );
NV_LUA_FUN( void ,(lua_setuservalue),(lua_State *L, int idx) );

/* 'load' and 'call' functions (load and run Lua code) */
NV_LUA_FUN( void ,(lua_callk),(lua_State *L, int nargs, int nresults, int ctx, lua_CFunction k) );
NV_LUA_FUN( int  ,(lua_getctx),(lua_State *L, int *ctx) );
NV_LUA_FUN( int  ,(lua_pcallk),(lua_State *L, int nargs, int nresults, int errfunc, int ctx, lua_CFunction k) );
NV_LUA_FUN( int  ,(lua_load),(lua_State *L, lua_Reader reader, void *dt, const char *chunkname, const char *mode) );
NV_LUA_FUN( int  ,(lua_dump),(lua_State *L, lua_Writer writer, void *data) );

/* Coroutine functions */
NV_LUA_FUN( int ,(lua_yieldk),(lua_State *L, int nresults, int ctx, lua_CFunction k) );
NV_LUA_FUN( int ,(lua_resume),(lua_State *L, lua_State *from, int narg) );
NV_LUA_FUN( int ,(lua_status),(lua_State *L) );

/* Garbage-collection function and options */
NV_LUA_FUN( int,(lua_gc),(lua_State *L, int what, int data) );

/* Miscellaneous functions */
NV_LUA_FUN( int  ,(lua_error),(lua_State *L) );
NV_LUA_FUN( int  ,(lua_next),(lua_State *L, int idx) );
NV_LUA_FUN( void ,(lua_concat),(lua_State *L, int n) );
NV_LUA_FUN( void ,(lua_len)   ,(lua_State *L, int idx) );
NV_LUA_FUN( lua_Alloc,(lua_getallocf),(lua_State *L, void **ud) );
NV_LUA_FUN( void     ,(lua_setallocf),(lua_State *L, lua_Alloc f, void *ud) );

/* Debug API */
NV_LUA_FUN( int,(lua_getstack),(lua_State *L, int level, lua_Debug *ar) );
NV_LUA_FUN( int,(lua_getinfo),(lua_State *L, const char *what, lua_Debug *ar) );
NV_LUA_FUN( const char *,(lua_getlocal),(lua_State *L, const lua_Debug *ar, int n) );
NV_LUA_FUN( const char *,(lua_setlocal),(lua_State *L, const lua_Debug *ar, int n) );
NV_LUA_FUN( const char *,(lua_getupvalue),(lua_State *L, int funcindex, int n) );
NV_LUA_FUN( const char *,(lua_setupvalue),(lua_State *L, int funcindex, int n) );
NV_LUA_FUN( void *,(lua_upvalueid),(lua_State *L, int fidx, int n) );
NV_LUA_FUN( void  ,(lua_upvaluejoin),(lua_State *L, int fidx1, int n1, int fidx2, int n2) );
NV_LUA_FUN( int,(lua_sethook),(lua_State *L, lua_Hook func, int mask, int count) );
NV_LUA_FUN( lua_Hook,(lua_gethook),(lua_State *L) );
NV_LUA_FUN( int,(lua_gethookmask),(lua_State *L) );
NV_LUA_FUN( int,(lua_gethookcount),(lua_State *L) );

/* lualib API */
NV_LUA_FUN( int,(luaopen_base),(lua_State *L) );
NV_LUA_FUN( int,(luaopen_coroutine),(lua_State *L) );
NV_LUA_FUN( int,(luaopen_table),(lua_State *L) );
NV_LUA_FUN( int,(luaopen_io),(lua_State *L) );
NV_LUA_FUN( int,(luaopen_os),(lua_State *L) );
NV_LUA_FUN( int,(luaopen_string),(lua_State *L) );
NV_LUA_FUN( int,(luaopen_bit32),(lua_State *L) );
NV_LUA_FUN( int,(luaopen_math),(lua_State *L) );
NV_LUA_FUN( int,(luaopen_debug),(lua_State *L) );
NV_LUA_FUN( int,(luaopen_package),(lua_State *L) );
NV_LUA_FUN( int,(luaL_openlibs),(lua_State *L) );

/* lauxlib API */
NV_LUA_FUN( void  ,(luaL_checkversion_),(lua_State *L, lua_Number ver) );
NV_LUA_FUN( int ,(luaL_getmetafield),(lua_State *L, int obj, const char *e) );
NV_LUA_FUN( int ,(luaL_callmeta),(lua_State *L, int obj, const char *e) );
NV_LUA_FUN( const char *,(luaL_tolstring),(lua_State *L, int idx, size_t *len) );
NV_LUA_FUN( int ,(luaL_argerror),(lua_State *L, int numarg, const char *extramsg) );
NV_LUA_FUN( const char *,(luaL_checklstring),(lua_State *L, int numArg, size_t *l) );
NV_LUA_FUN( const char *,(luaL_optlstring),(lua_State *L, int numArg, const char *def, size_t *l) );
NV_LUA_FUN( lua_Number ,(luaL_checknumber),(lua_State *L, int numArg) );
NV_LUA_FUN( lua_Number ,(luaL_optnumber),(lua_State *L, int nArg, lua_Number def) );
NV_LUA_FUN( lua_Integer ,(luaL_checkinteger),(lua_State *L, int numArg) );
NV_LUA_FUN( lua_Integer ,(luaL_optinteger),(lua_State *L, int nArg, lua_Integer def) );
NV_LUA_FUN( lua_Unsigned ,(luaL_checkunsigned),(lua_State *L, int numArg) );
NV_LUA_FUN( lua_Unsigned ,(luaL_optunsigned),(lua_State *L, int numArg, lua_Unsigned def) );
NV_LUA_FUN( void ,(luaL_checkstack),(lua_State *L, int sz, const char *msg) );
NV_LUA_FUN( void ,(luaL_checktype),(lua_State *L, int narg, int t) );
NV_LUA_FUN( void ,(luaL_checkany),(lua_State *L, int narg) );
NV_LUA_FUN( int   ,(luaL_newmetatable),(lua_State *L, const char *tname) );
NV_LUA_FUN( void  ,(luaL_setmetatable),(lua_State *L, const char *tname) );
NV_LUA_FUN( void *,(luaL_testudata),(lua_State *L, int ud, const char *tname) );
NV_LUA_FUN( void *,(luaL_checkudata),(lua_State *L, int ud, const char *tname) );
NV_LUA_FUN( void ,(luaL_where),(lua_State *L, int lvl) );
NV_LUA_FUN( int ,(luaL_error),(lua_State *L, const char *fmt, ...) );
NV_LUA_FUN( int ,(luaL_checkoption),(lua_State *L, int narg, const char *def, const char *const lst[]) );
NV_LUA_FUN( int ,(luaL_fileresult),(lua_State *L, int stat, const char *fname) );
NV_LUA_FUN( int ,(luaL_execresult),(lua_State *L, int stat) );

NV_LUA_FUN( int ,(luaL_ref),(lua_State *L, int t) );
NV_LUA_FUN( void ,(luaL_unref),(lua_State *L, int t, int ref) );
NV_LUA_FUN( int ,(luaL_loadfilex),(lua_State *L, const char *filename, const char *mode) );
NV_LUA_FUN( int ,(luaL_loadbufferx),(lua_State *L, const char *buff, size_t sz, const char *name, const char *mode) );
NV_LUA_FUN( int ,(luaL_loadstring),(lua_State *L, const char *s) );
NV_LUA_FUN( lua_State *,(luaL_newstate), (void) );
NV_LUA_FUN( int ,(luaL_len),(lua_State *L, int idx) );
NV_LUA_FUN( const char *,(luaL_gsub),(lua_State *L, const char *s, const char *p, const char *r) );
NV_LUA_FUN( void ,(luaL_setfuncs),(lua_State *L, const luaL_Reg *l, int nup) );
NV_LUA_FUN( int ,(luaL_getsubtable),(lua_State *L, int idx, const char *fname) );
NV_LUA_FUN( void ,(luaL_traceback),(lua_State *L, lua_State *L1, const char *msg, int level) );
NV_LUA_FUN( void ,(luaL_requiref),(lua_State *L, const char *modname, lua_CFunction openf, int glb) );

#undef NV_LUA_FUN

/* Macros */
#define lua_call(L,n,r)		lua_callk(L, (n), (r), 0, NULL)
#define lua_pcall(L,n,r,f)	lua_pcallk(L, (n), (r), (f), 0, NULL)
#define lua_yield(L,n)		lua_yieldk(L, (n), 0, NULL)
#define lua_tonumber(L,i)	lua_tonumberx(L,i,NULL)
#define lua_tointeger(L,i)	lua_tointegerx(L,i,NULL)
#define lua_tounsigned(L,i)	lua_tounsignedx(L,i,NULL)
#define lua_pop(L,n)		lua_settop(L, -(n)-1)
#define lua_newtable(L)		lua_createtable(L, 0, 0)
#define lua_register(L,n,f) (lua_pushcfunction(L, (f)), lua_setglobal(L, (n)))
#define lua_pushcfunction(L,f)	lua_pushcclosure(L, (f), 0)
#define lua_isfunction(L,n)	(lua_type(L, (n)) == LUA_TFUNCTION)
#define lua_istable(L,n)	(lua_type(L, (n)) == LUA_TTABLE)
#define lua_islightuserdata(L,n)	(lua_type(L, (n)) == LUA_TLIGHTUSERDATA)
#define lua_isnil(L,n)		(lua_type(L, (n)) == LUA_TNIL)
#define lua_isboolean(L,n)	(lua_type(L, (n)) == LUA_TBOOLEAN)
#define lua_isthread(L,n)	(lua_type(L, (n)) == LUA_TTHREAD)
#define lua_isnone(L,n)		(lua_type(L, (n)) == LUA_TNONE)
#define lua_isnoneornil(L, n)	(lua_type(L, (n)) <= 0)
#define lua_pushliteral(L, s)	lua_pushlstring(L, "" s, (sizeof(s)/sizeof(char))-1)
#define lua_pushglobaltable(L)  lua_rawgeti(L, LUA_REGISTRYINDEX, LUA_RIDX_GLOBALS)
#define lua_tostring(L,i)	lua_tolstring(L, (i), NULL)

/* Aux lib macros */
#define luaL_checkversion(L)	luaL_checkversion_(L, LUA_VERSION_NUM)
#define luaL_loadfile(L,f)	luaL_loadfilex(L,f,NULL)
#define luaL_newlibtable(L,l)	lua_createtable(L, 0, sizeof(l)/sizeof((l)[0]) - 1)
#define luaL_newlib(L,l)	(luaL_newlibtable(L,l), luaL_setfuncs(L,l,0))
#define luaL_argcheck(L, cond,numarg,extramsg)	((void)((cond) || luaL_argerror(L, (numarg), (extramsg))))
#define luaL_checkstring(L,n)	(luaL_checklstring(L, (n), NULL))
#define luaL_optstring(L,n,d)	(luaL_optlstring(L, (n), (d), NULL))
#define luaL_checkint(L,n)	((int)luaL_checkinteger(L, (n)))
#define luaL_optint(L,n,d)	((int)luaL_optinteger(L, (n), (d)))
#define luaL_checklong(L,n)	((long)luaL_checkinteger(L, (n)))
#define luaL_optlong(L,n,d)	((long)luaL_optinteger(L, (n), (d)))
#define luaL_typename(L,i)	lua_typename(L, lua_type(L,(i)))
#define luaL_dofile(L, fn) (luaL_loadfile(L, fn) || lua_pcall(L, 0, LUA_MULTRET, 0))
#define luaL_dostring(L, s) (luaL_loadstring(L, s) || lua_pcall(L, 0, LUA_MULTRET, 0))
#define luaL_getmetatable(L,n)	(lua_getfield(L, LUA_REGISTRYINDEX, (n)))
#define luaL_opt(L,f,n,d)	(lua_isnoneornil(L,(n)) ? (d) : f(L,(n)))
#define luaL_loadbuffer(L,s,sz,n)	luaL_loadbufferx(L,s,sz,n,NULL)
#define luaL_addchar(B,c)  ((void)((B)->n < (B)->size || luaL_prepbuffsize((B), 1)), ((B)->b[(B)->n++] = (c)))
#define luaL_addsize(B,s)	((B)->n += (s))
#define luaL_prepbuffer(B)	luaL_prepbuffsize(B, LUAL_BUFFERSIZE)

}

namespace nv {
/* Dynamic load support */
#	if defined( NV_LUA_DYNAMIC )
		bool load_lua_library( const char* path = NV_LUA_PATH );
#	else
		inline bool load_lua_library( const char* path = "" ) { return true; }
#	endif
}

#endif // NV_LIB_LUA_HH
