// Copyright (C) 2012-2013 ChaosForge / Kornel Kisielewicz
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

#ifndef NV_LIB_FREETYPE_HH
#define NV_LIB_FREETYPE_HH

#include <stdint.h>

#define NV_FREETYPE_DYNAMIC
//#define NV_FREETYPE_SHARED

#if NV_PLATFORM == NV_WINDOWS
#	define NV_FREETYPE_PATH "freetype6.dll"
#elif NV_PLATFORM == NV_APPLE
#	define NV_FREETYPE_PATH "freetype.framework/freetype"
#else
#	define NV_FREETYPE_PATH "libfreetype.so"
#endif

extern "C" {

#if NV_PLATFORM == NV_WINDOWS 
#	define NV_FREETYPE_APIENTRY __cdecl
#else
#	define NV_FREETYPE_APIENTRY 
#endif

#if defined(NV_FREETYPE_SHARED) && (NV_PLATFORM == NV_WINDOWS)
#	define NV_FREETYPE_API __declspec(dllimport)
#else
#	define NV_FREETYPE_API extern
#endif

#if defined(NV_FREETYPE_DYNAMIC)
#	define NV_FREETYPE_FUN( rtype, fname, fparams ) NV_FREETYPE_API rtype (NV_FREETYPE_APIENTRY *fname) fparams
#else
#	define NV_FREETYPE_FUN( rtype, fname, fparams ) NV_FREETYPE_API rtype NV_FREETYPE_APIENTRY fname fparams
#endif

#define FREETYPE_MAJOR  2
#define FREETYPE_MINOR  4
#define FREETYPE_PATCH  2

  typedef signed short    FT_Int16;
  typedef unsigned short  FT_UInt16;
  typedef signed int      FT_Int32;
  typedef unsigned int    FT_UInt32;
  typedef int             FT_Fast;
  typedef unsigned int    FT_UFast;

  /* fttypes.h */
  typedef unsigned char  FT_Bool;
  typedef signed short  FT_FWord; 
  typedef unsigned short  FT_UFWord; 
  typedef signed char  FT_Char;
  typedef unsigned char  FT_Byte;
  typedef const FT_Byte*  FT_Bytes;
  typedef FT_UInt32  FT_Tag;
  typedef char  FT_String;
  typedef signed short  FT_Short;
  typedef unsigned short  FT_UShort;
  typedef signed int  FT_Int;
  typedef unsigned int  FT_UInt;
  typedef signed long  FT_Long;
  typedef unsigned long  FT_ULong;
  typedef signed short  FT_F2Dot14;
  typedef signed long  FT_F26Dot6;
  typedef signed long  FT_Fixed;
  typedef int  FT_Error;
  typedef void*  FT_Pointer;
  typedef size_t  FT_Offset;
  typedef ptrdiff_t  FT_PtrDist;

  typedef struct  FT_UnitVector_
  {
    FT_F2Dot14  x;
    FT_F2Dot14  y;

  } FT_UnitVector;

  typedef struct  FT_Matrix_
  {
    FT_Fixed  xx, xy;
    FT_Fixed  yx, yy;

  } FT_Matrix;

  typedef struct  FT_Data_
  {
    const FT_Byte*  pointer;
    FT_Int          length;

  } FT_Data;

  typedef void  (*FT_Generic_Finalizer)(void*  object);

  typedef struct  FT_Generic_
  {
    void*                 data;
    FT_Generic_Finalizer  finalizer;

  } FT_Generic;

#define FT_MAKE_TAG( _x1, _x2, _x3, _x4 ) \
          (FT_Tag)                        \
          ( ( (FT_ULong)_x1 << 24 ) |     \
            ( (FT_ULong)_x2 << 16 ) |     \
            ( (FT_ULong)_x3 <<  8 ) |     \
              (FT_ULong)_x4         )

  /* ftimage.h */

  typedef signed long  FT_Pos;

  typedef struct  FT_Vector_
  {
    FT_Pos  x;
    FT_Pos  y;

  } FT_Vector;

  typedef struct  FT_BBox_
  {
    FT_Pos  xMin, yMin;
    FT_Pos  xMax, yMax;

  } FT_BBox;

  typedef enum  FT_Pixel_Mode_
  {
    FT_PIXEL_MODE_NONE = 0,
    FT_PIXEL_MODE_MONO,
    FT_PIXEL_MODE_GRAY,
    FT_PIXEL_MODE_GRAY2,
    FT_PIXEL_MODE_GRAY4,
    FT_PIXEL_MODE_LCD,
    FT_PIXEL_MODE_LCD_V,

    FT_PIXEL_MODE_MAX      /* do not remove */

  } FT_Pixel_Mode;

  typedef struct  FT_Bitmap_
  {
    int             rows;
    int             width;
    int             pitch;
    unsigned char*  buffer;
    short           num_grays;
    char            pixel_mode;
    char            palette_mode;
    void*           palette;

  } FT_Bitmap;

  typedef struct  FT_Outline_
  {
    short       n_contours;      /* number of contours in glyph        */
    short       n_points;        /* number of points in the glyph      */

    FT_Vector*  points;          /* the outline's points               */
    char*       tags;            /* the points flags                   */
    short*      contours;        /* the contour end points             */

    int         flags;           /* outline masks                      */

  } FT_Outline;

#define FT_OUTLINE_CONTOURS_MAX  SHRT_MAX
#define FT_OUTLINE_POINTS_MAX    SHRT_MAX
#define FT_OUTLINE_NONE             0x0
#define FT_OUTLINE_OWNER            0x1
#define FT_OUTLINE_EVEN_ODD_FILL    0x2
#define FT_OUTLINE_REVERSE_FILL     0x4
#define FT_OUTLINE_IGNORE_DROPOUTS  0x8
#define FT_OUTLINE_SMART_DROPOUTS   0x10
#define FT_OUTLINE_INCLUDE_STUBS    0x20

#define FT_OUTLINE_HIGH_PRECISION   0x100
#define FT_OUTLINE_SINGLE_PASS      0x200

#define FT_CURVE_TAG( flag )  ( flag & 3 )

#define FT_CURVE_TAG_ON            1
#define FT_CURVE_TAG_CONIC         0
#define FT_CURVE_TAG_CUBIC         2

#define FT_CURVE_TAG_HAS_SCANMODE  4

#define FT_CURVE_TAG_TOUCH_X       8  /* reserved for the TrueType hinter */
#define FT_CURVE_TAG_TOUCH_Y      16  /* reserved for the TrueType hinter */

#define FT_CURVE_TAG_TOUCH_BOTH    ( FT_CURVE_TAG_TOUCH_X | \
                                     FT_CURVE_TAG_TOUCH_Y )

  typedef int (*FT_Outline_MoveToFunc)( const FT_Vector*  to, void* user );
  typedef int (*FT_Outline_LineToFunc)( const FT_Vector*  to, void* user );
  typedef int (*FT_Outline_ConicToFunc)( const FT_Vector*  control, const FT_Vector*  to, void* user );
  typedef int (*FT_Outline_CubicToFunc)( const FT_Vector*  control1, const FT_Vector*  control2, const FT_Vector*  to, void* user );

  typedef struct  FT_Outline_Funcs_
  {
    FT_Outline_MoveToFunc   move_to;
    FT_Outline_LineToFunc   line_to;
    FT_Outline_ConicToFunc  conic_to;
    FT_Outline_CubicToFunc  cubic_to;

    int                     shift;
    FT_Pos                  delta;

  } FT_Outline_Funcs;

#ifndef FT_IMAGE_TAG
#define FT_IMAGE_TAG( value, _x1, _x2, _x3, _x4 )  \
          value = ( ( (unsigned long)_x1 << 24 ) | \
                    ( (unsigned long)_x2 << 16 ) | \
                    ( (unsigned long)_x3 << 8  ) | \
                      (unsigned long)_x4         )
#endif /* FT_IMAGE_TAG */

  typedef enum  FT_Glyph_Format_
  {
    FT_IMAGE_TAG( FT_GLYPH_FORMAT_NONE, 0, 0, 0, 0 ),

    FT_IMAGE_TAG( FT_GLYPH_FORMAT_COMPOSITE, 'c', 'o', 'm', 'p' ),
    FT_IMAGE_TAG( FT_GLYPH_FORMAT_BITMAP,    'b', 'i', 't', 's' ),
    FT_IMAGE_TAG( FT_GLYPH_FORMAT_OUTLINE,   'o', 'u', 't', 'l' ),
    FT_IMAGE_TAG( FT_GLYPH_FORMAT_PLOTTER,   'p', 'l', 'o', 't' )

  } FT_Glyph_Format;

  typedef struct FT_RasterRec_*  FT_Raster;

  typedef struct  FT_Span_
  {
    short           x;
    unsigned short  len;
    unsigned char   coverage;

  } FT_Span;

  typedef void (*FT_SpanFunc)( int y, int count, const FT_Span* spans, void* user );

#define FT_RASTER_FLAG_DEFAULT  0x0
#define FT_RASTER_FLAG_AA       0x1
#define FT_RASTER_FLAG_DIRECT   0x2
#define FT_RASTER_FLAG_CLIP     0x4

  typedef struct  FT_Raster_Params_
  {
    const FT_Bitmap*        target;
    const void*             source;
    int                     flags;
    FT_SpanFunc             gray_spans;
    FT_SpanFunc             black_spans;
    FT_SpanFunc             bit_test;
    FT_SpanFunc             bit_set;
    void*                   user;
    FT_BBox                 clip_box;

  } FT_Raster_Params;

  typedef int (*FT_Raster_NewFunc)( void* memory, FT_Raster* raster );
  typedef void (*FT_Raster_DoneFunc)( FT_Raster raster );
  typedef void (*FT_Raster_ResetFunc)( FT_Raster raster, unsigned char* pool_base, unsigned long pool_size );
  typedef int (*FT_Raster_SetModeFunc)( FT_Raster raster, unsigned long mode, void* args );
  typedef int (*FT_Raster_RenderFunc)( FT_Raster raster, const FT_Raster_Params*  params );

  typedef struct  FT_Raster_Funcs_
  {
    FT_Glyph_Format        glyph_format;
    FT_Raster_NewFunc      raster_new;
    FT_Raster_ResetFunc    raster_reset;
    FT_Raster_SetModeFunc  raster_set_mode;
    FT_Raster_RenderFunc   raster_render;
    FT_Raster_DoneFunc     raster_done;

  } FT_Raster_Funcs;

  typedef struct FT_ListNodeRec_*  FT_ListNode;

  typedef struct FT_ListRec_*  FT_List;

  typedef struct  FT_ListNodeRec_
  {
    FT_ListNode  prev;
    FT_ListNode  next;
    void*        data;

  } FT_ListNodeRec;

  typedef struct  FT_ListRec_
  {
    FT_ListNode  head;
    FT_ListNode  tail;

  } FT_ListRec;

 

  /* ftsystem.h */
  typedef struct FT_MemoryRec_*  FT_Memory;
  typedef void* (*FT_Alloc_Func)( FT_Memory memory, long size );
  typedef void (*FT_Free_Func)( FT_Memory memory, void* block );
  typedef void* (*FT_Realloc_Func)( FT_Memory memory, long cur_size, long new_size, void* block );

  struct  FT_MemoryRec_
  {
    void*            user;
    FT_Alloc_Func    alloc;
    FT_Free_Func     free;
    FT_Realloc_Func  realloc;
  };

  typedef struct FT_StreamRec_*  FT_Stream;

  typedef union  FT_StreamDesc_
  {
    long   value;
    void*  pointer;

  } FT_StreamDesc;

  typedef unsigned long (*FT_Stream_IoFunc)( FT_Stream stream, unsigned long   offset, unsigned char*  buffer, unsigned long count );
  typedef void (*FT_Stream_CloseFunc)( FT_Stream  stream );

  typedef struct  FT_StreamRec_
  {
    unsigned char*       base;
    unsigned long        size;
    unsigned long        pos;

    FT_StreamDesc        descriptor;
    FT_StreamDesc        pathname;
    FT_Stream_IoFunc     read;
    FT_Stream_CloseFunc  close;

    FT_Memory            memory;
    unsigned char*       cursor;
    unsigned char*       limit;

  } FT_StreamRec; 

  /* freetype.h */

  typedef struct  FT_Glyph_Metrics_
  {
    FT_Pos  width;
    FT_Pos  height;

    FT_Pos  horiBearingX;
    FT_Pos  horiBearingY;
    FT_Pos  horiAdvance;

    FT_Pos  vertBearingX;
    FT_Pos  vertBearingY;
    FT_Pos  vertAdvance;

  } FT_Glyph_Metrics;

  typedef struct  FT_Bitmap_Size_
  {
    FT_Short  height;
    FT_Short  width;

    FT_Pos    size;

    FT_Pos    x_ppem;
    FT_Pos    y_ppem;

  } FT_Bitmap_Size;

  typedef struct FT_LibraryRec_  *FT_Library;
  typedef struct FT_ModuleRec_*  FT_Module;
  typedef struct FT_DriverRec_*  FT_Driver;
  typedef struct FT_RendererRec_*  FT_Renderer;
  typedef struct FT_FaceRec_*  FT_Face;
  typedef struct FT_SizeRec_*  FT_Size;
  typedef struct FT_GlyphSlotRec_*  FT_GlyphSlot;
  typedef struct FT_CharMapRec_*  FT_CharMap;

#ifndef FT_ENC_TAG
#define FT_ENC_TAG( value, a, b, c, d )         \
          value = ( ( (FT_UInt32)(a) << 24 ) |  \
                    ( (FT_UInt32)(b) << 16 ) |  \
                    ( (FT_UInt32)(c) <<  8 ) |  \
                      (FT_UInt32)(d)         )

#endif /* FT_ENC_TAG */

  typedef enum  FT_Encoding_
  {
    FT_ENC_TAG( FT_ENCODING_NONE, 0, 0, 0, 0 ),

    FT_ENC_TAG( FT_ENCODING_MS_SYMBOL, 's', 'y', 'm', 'b' ),
    FT_ENC_TAG( FT_ENCODING_UNICODE,   'u', 'n', 'i', 'c' ),

    FT_ENC_TAG( FT_ENCODING_SJIS,    's', 'j', 'i', 's' ),
    FT_ENC_TAG( FT_ENCODING_GB2312,  'g', 'b', ' ', ' ' ),
    FT_ENC_TAG( FT_ENCODING_BIG5,    'b', 'i', 'g', '5' ),
    FT_ENC_TAG( FT_ENCODING_WANSUNG, 'w', 'a', 'n', 's' ),
    FT_ENC_TAG( FT_ENCODING_JOHAB,   'j', 'o', 'h', 'a' ),

    /* for backwards compatibility */
    FT_ENCODING_MS_SJIS    = FT_ENCODING_SJIS,
    FT_ENCODING_MS_GB2312  = FT_ENCODING_GB2312,
    FT_ENCODING_MS_BIG5    = FT_ENCODING_BIG5,
    FT_ENCODING_MS_WANSUNG = FT_ENCODING_WANSUNG,
    FT_ENCODING_MS_JOHAB   = FT_ENCODING_JOHAB,

    FT_ENC_TAG( FT_ENCODING_ADOBE_STANDARD, 'A', 'D', 'O', 'B' ),
    FT_ENC_TAG( FT_ENCODING_ADOBE_EXPERT,   'A', 'D', 'B', 'E' ),
    FT_ENC_TAG( FT_ENCODING_ADOBE_CUSTOM,   'A', 'D', 'B', 'C' ),
    FT_ENC_TAG( FT_ENCODING_ADOBE_LATIN_1,  'l', 'a', 't', '1' ),

    FT_ENC_TAG( FT_ENCODING_OLD_LATIN_2, 'l', 'a', 't', '2' ),

    FT_ENC_TAG( FT_ENCODING_APPLE_ROMAN, 'a', 'r', 'm', 'n' )

  } FT_Encoding;

  typedef struct  FT_CharMapRec_
  {
    FT_Face      face;
    FT_Encoding  encoding;
    FT_UShort    platform_id;
    FT_UShort    encoding_id;

  } FT_CharMapRec;

  typedef struct FT_Face_InternalRec_*  FT_Face_Internal;

  typedef struct  FT_FaceRec_
  {
    FT_Long           num_faces;
    FT_Long           face_index;

    FT_Long           face_flags;
    FT_Long           style_flags;

    FT_Long           num_glyphs;

    FT_String*        family_name;
    FT_String*        style_name;

    FT_Int            num_fixed_sizes;
    FT_Bitmap_Size*   available_sizes;

    FT_Int            num_charmaps;
    FT_CharMap*       charmaps;

    FT_Generic        generic;

    FT_BBox           bbox;

    FT_UShort         units_per_EM;
    FT_Short          ascender;
    FT_Short          descender;
    FT_Short          height;

    FT_Short          max_advance_width;
    FT_Short          max_advance_height;

    FT_Short          underline_position;
    FT_Short          underline_thickness;

    FT_GlyphSlot      glyph;
    FT_Size           size;
    FT_CharMap        charmap;

    /*@private begin */

    FT_Driver         driver;
    FT_Memory         memory;
    FT_Stream         stream;

    FT_ListRec        sizes_list;

    FT_Generic        autohint;
    void*             extensions;

    FT_Face_Internal  internal;

  } FT_FaceRec;

#define FT_FACE_FLAG_SCALABLE          ( 1L <<  0 )
#define FT_FACE_FLAG_FIXED_SIZES       ( 1L <<  1 )
#define FT_FACE_FLAG_FIXED_WIDTH       ( 1L <<  2 )
#define FT_FACE_FLAG_SFNT              ( 1L <<  3 )
#define FT_FACE_FLAG_HORIZONTAL        ( 1L <<  4 )
#define FT_FACE_FLAG_VERTICAL          ( 1L <<  5 )
#define FT_FACE_FLAG_KERNING           ( 1L <<  6 )
#define FT_FACE_FLAG_FAST_GLYPHS       ( 1L <<  7 )
#define FT_FACE_FLAG_MULTIPLE_MASTERS  ( 1L <<  8 )
#define FT_FACE_FLAG_GLYPH_NAMES       ( 1L <<  9 )
#define FT_FACE_FLAG_EXTERNAL_STREAM   ( 1L << 10 )
#define FT_FACE_FLAG_HINTER            ( 1L << 11 )
#define FT_FACE_FLAG_CID_KEYED         ( 1L << 12 )
#define FT_FACE_FLAG_TRICKY            ( 1L << 13 )

#define FT_HAS_HORIZONTAL( face ) \
          ( face->face_flags & FT_FACE_FLAG_HORIZONTAL )

#define FT_HAS_VERTICAL( face ) \
          ( face->face_flags & FT_FACE_FLAG_VERTICAL )

#define FT_HAS_KERNING( face ) \
          ( face->face_flags & FT_FACE_FLAG_KERNING )

#define FT_IS_SCALABLE( face ) \
          ( face->face_flags & FT_FACE_FLAG_SCALABLE )

#define FT_IS_SFNT( face ) \
          ( face->face_flags & FT_FACE_FLAG_SFNT )

#define FT_IS_FIXED_WIDTH( face ) \
          ( face->face_flags & FT_FACE_FLAG_FIXED_WIDTH )

#define FT_HAS_FIXED_SIZES( face ) \
          ( face->face_flags & FT_FACE_FLAG_FIXED_SIZES )

#define FT_HAS_FAST_GLYPHS( face )  0

#define FT_HAS_GLYPH_NAMES( face ) \
          ( face->face_flags & FT_FACE_FLAG_GLYPH_NAMES )

#define FT_HAS_MULTIPLE_MASTERS( face ) \
          ( face->face_flags & FT_FACE_FLAG_MULTIPLE_MASTERS )

#define FT_IS_CID_KEYED( face ) \
          ( face->face_flags & FT_FACE_FLAG_CID_KEYED )

#define FT_IS_TRICKY( face ) \
          ( face->face_flags & FT_FACE_FLAG_TRICKY )

#define FT_STYLE_FLAG_ITALIC  ( 1 << 0 )
#define FT_STYLE_FLAG_BOLD    ( 1 << 1 )

  typedef struct FT_Size_InternalRec_*  FT_Size_Internal;

  typedef struct  FT_Size_Metrics_
  {
    FT_UShort  x_ppem;      /* horizontal pixels per EM               */
    FT_UShort  y_ppem;      /* vertical pixels per EM                 */

    FT_Fixed   x_scale;     /* scaling values used to convert font    */
    FT_Fixed   y_scale;     /* units to 26.6 fractional pixels        */

    FT_Pos     ascender;    /* ascender in 26.6 frac. pixels          */
    FT_Pos     descender;   /* descender in 26.6 frac. pixels         */
    FT_Pos     height;      /* text height in 26.6 frac. pixels       */
    FT_Pos     max_advance; /* max horizontal advance, in 26.6 pixels */

  } FT_Size_Metrics;

  typedef struct  FT_SizeRec_
  {
    FT_Face           face;      /* parent face object              */
    FT_Generic        generic;   /* generic pointer for client uses */
    FT_Size_Metrics   metrics;   /* size metrics                    */
    FT_Size_Internal  internal;

  } FT_SizeRec;

  typedef struct FT_SubGlyphRec_*  FT_SubGlyph;

  typedef struct FT_Slot_InternalRec_*  FT_Slot_Internal;

  typedef struct  FT_GlyphSlotRec_
  {
    FT_Library        library;
    FT_Face           face;
    FT_GlyphSlot      next;
    FT_UInt           reserved;       /* retained for binary compatibility */
    FT_Generic        generic;

    FT_Glyph_Metrics  metrics;
    FT_Fixed          linearHoriAdvance;
    FT_Fixed          linearVertAdvance;
    FT_Vector         advance;

    FT_Glyph_Format   format;

    FT_Bitmap         bitmap;
    FT_Int            bitmap_left;
    FT_Int            bitmap_top;

    FT_Outline        outline;

    FT_UInt           num_subglyphs;
    FT_SubGlyph       subglyphs;

    void*             control_data;
    long              control_len;

    FT_Pos            lsb_delta;
    FT_Pos            rsb_delta;

    void*             other;

    FT_Slot_Internal  internal;

  } FT_GlyphSlotRec;

#define FT_OPEN_MEMORY    0x1
#define FT_OPEN_STREAM    0x2
#define FT_OPEN_PATHNAME  0x4
#define FT_OPEN_DRIVER    0x8
#define FT_OPEN_PARAMS    0x10

  typedef struct  FT_Parameter_
  {
    FT_ULong    tag;
    FT_Pointer  data;

  } FT_Parameter;

  typedef struct  FT_Open_Args_
  {
    FT_UInt         flags;
    const FT_Byte*  memory_base;
    FT_Long         memory_size;
    FT_String*      pathname;
    FT_Stream       stream;
    FT_Module       driver;
    FT_Int          num_params;
    FT_Parameter*   params;

  } FT_Open_Args;

  typedef enum  FT_Size_Request_Type_
  {
    FT_SIZE_REQUEST_TYPE_NOMINAL,
    FT_SIZE_REQUEST_TYPE_REAL_DIM,
    FT_SIZE_REQUEST_TYPE_BBOX,
    FT_SIZE_REQUEST_TYPE_CELL,
    FT_SIZE_REQUEST_TYPE_SCALES,

    FT_SIZE_REQUEST_TYPE_MAX

  } FT_Size_Request_Type;

  typedef struct  FT_Size_RequestRec_
  {
    FT_Size_Request_Type  type;
    FT_Long               width;
    FT_Long               height;
    FT_UInt               horiResolution;
    FT_UInt               vertResolution;

  } FT_Size_RequestRec;

  typedef struct FT_Size_RequestRec_  *FT_Size_Request;

#define FT_LOAD_DEFAULT                      0x0
#define FT_LOAD_NO_SCALE                     0x1
#define FT_LOAD_NO_HINTING                   0x2
#define FT_LOAD_RENDER                       0x4
#define FT_LOAD_NO_BITMAP                    0x8
#define FT_LOAD_VERTICAL_LAYOUT              0x10
#define FT_LOAD_FORCE_AUTOHINT               0x20
#define FT_LOAD_CROP_BITMAP                  0x40
#define FT_LOAD_PEDANTIC                     0x80
#define FT_LOAD_IGNORE_GLOBAL_ADVANCE_WIDTH  0x200
#define FT_LOAD_NO_RECURSE                   0x400
#define FT_LOAD_IGNORE_TRANSFORM             0x800
#define FT_LOAD_MONOCHROME                   0x1000
#define FT_LOAD_LINEAR_DESIGN                0x2000
#define FT_LOAD_NO_AUTOHINT                  0x8000U
#define FT_LOAD_ADVANCE_ONLY                 0x100
#define FT_LOAD_SBITS_ONLY                   0x4000
#define FT_LOAD_TARGET_( x )   ( (FT_Int32)( (x) & 15 ) << 16 )
#define FT_LOAD_TARGET_NORMAL  FT_LOAD_TARGET_( FT_RENDER_MODE_NORMAL )
#define FT_LOAD_TARGET_LIGHT   FT_LOAD_TARGET_( FT_RENDER_MODE_LIGHT  )
#define FT_LOAD_TARGET_MONO    FT_LOAD_TARGET_( FT_RENDER_MODE_MONO   )
#define FT_LOAD_TARGET_LCD     FT_LOAD_TARGET_( FT_RENDER_MODE_LCD    )
#define FT_LOAD_TARGET_LCD_V   FT_LOAD_TARGET_( FT_RENDER_MODE_LCD_V  )
#define FT_LOAD_TARGET_MODE( x )  ( (FT_Render_Mode)( ( (x) >> 16 ) & 15 ) )

  typedef enum  FT_Render_Mode_
  {
    FT_RENDER_MODE_NORMAL = 0,
    FT_RENDER_MODE_LIGHT,
    FT_RENDER_MODE_MONO,
    FT_RENDER_MODE_LCD,
    FT_RENDER_MODE_LCD_V,

    FT_RENDER_MODE_MAX

  } FT_Render_Mode;

  typedef enum  FT_Kerning_Mode_
  {
    FT_KERNING_DEFAULT  = 0,
    FT_KERNING_UNFITTED,
    FT_KERNING_UNSCALED

  } FT_Kerning_Mode;

#define FT_SUBGLYPH_FLAG_ARGS_ARE_WORDS          1
#define FT_SUBGLYPH_FLAG_ARGS_ARE_XY_VALUES      2
#define FT_SUBGLYPH_FLAG_ROUND_XY_TO_GRID        4
#define FT_SUBGLYPH_FLAG_SCALE                   8
#define FT_SUBGLYPH_FLAG_XY_SCALE             0x40
#define FT_SUBGLYPH_FLAG_2X2                  0x80
#define FT_SUBGLYPH_FLAG_USE_MY_METRICS      0x200

#define FT_FSTYPE_INSTALLABLE_EMBEDDING         0x0000
#define FT_FSTYPE_RESTRICTED_LICENSE_EMBEDDING  0x0002
#define FT_FSTYPE_PREVIEW_AND_PRINT_EMBEDDING   0x0004
#define FT_FSTYPE_EDITABLE_EMBEDDING            0x0008
#define FT_FSTYPE_NO_SUBSETTING                 0x0100
#define FT_FSTYPE_BITMAP_EMBEDDING_ONLY         0x0200

  /* ftgylph.h */

  typedef struct FT_Glyph_Class_  FT_Glyph_Class;
  typedef struct FT_GlyphRec_*  FT_Glyph;
  typedef struct  FT_GlyphRec_
  {
    FT_Library             library;
    const FT_Glyph_Class*  clazz;
    FT_Glyph_Format        format;
    FT_Vector              advance;

  } FT_GlyphRec;

  typedef struct FT_BitmapGlyphRec_*  FT_BitmapGlyph;

  typedef struct  FT_BitmapGlyphRec_
  {
    FT_GlyphRec  root;
    FT_Int       left;
    FT_Int       top;
    FT_Bitmap    bitmap;

  } FT_BitmapGlyphRec;

  typedef struct FT_OutlineGlyphRec_*  FT_OutlineGlyph;

  typedef struct  FT_OutlineGlyphRec_
  {
    FT_GlyphRec  root;
    FT_Outline   outline;

  } FT_OutlineGlyphRec;

  typedef enum  FT_Glyph_BBox_Mode_
  {
    FT_GLYPH_BBOX_UNSCALED  = 0,
    FT_GLYPH_BBOX_SUBPIXELS = 0,
    FT_GLYPH_BBOX_GRIDFIT   = 1,
    FT_GLYPH_BBOX_TRUNCATE  = 2,
    FT_GLYPH_BBOX_PIXELS    = 3

  } FT_Glyph_BBox_Mode;

  typedef enum  FT_LcdFilter_
  {
    FT_LCD_FILTER_NONE    = 0,
    FT_LCD_FILTER_DEFAULT = 1,
    FT_LCD_FILTER_LIGHT   = 2,
    FT_LCD_FILTER_LEGACY  = 16,

    FT_LCD_FILTER_MAX   /* do not remove */

  } FT_LcdFilter; 

  /* ftmodapi.h */

#define FT_MODULE_FONT_DRIVER         1  /* this module is a font driver  */
#define FT_MODULE_RENDERER            2  /* this module is a renderer     */
#define FT_MODULE_HINTER              4  /* this module is a glyph hinter */
#define FT_MODULE_STYLER              8  /* this module is a styler       */

#define FT_MODULE_DRIVER_SCALABLE     0x100   /* the driver supports      */
                                              /* scalable fonts           */
#define FT_MODULE_DRIVER_NO_OUTLINES  0x200   /* the driver does not      */
                                              /* support vector outlines  */
#define FT_MODULE_DRIVER_HAS_HINTER   0x400   /* the driver provides its  */
                                              /* own hinter               */

  typedef FT_Pointer  FT_Module_Interface;

  typedef FT_Error
  (*FT_Module_Constructor)( FT_Module  module );

  typedef void
  (*FT_Module_Destructor)( FT_Module  module );

  typedef FT_Module_Interface
  (*FT_Module_Requester)( FT_Module    module,
                          const char*  name );

  typedef struct  FT_Module_Class_
  {
    FT_ULong               module_flags;
    FT_Long                module_size;
    const FT_String*       module_name;
    FT_Fixed               module_version;
    FT_Fixed               module_requires;

    const void*            module_interface;

    FT_Module_Constructor  module_init;
    FT_Module_Destructor   module_done;
    FT_Module_Requester    get_interface;

  } FT_Module_Class;

  typedef enum  FT_TrueTypeEngineType_
  {
    FT_TRUETYPE_ENGINE_TYPE_NONE = 0,
    FT_TRUETYPE_ENGINE_TYPE_UNPATENTED,
    FT_TRUETYPE_ENGINE_TYPE_PATENTED

  } FT_TrueTypeEngineType;

  typedef void
  (*FT_DebugHook_Func)( void*  arg );


  /* ftrender.h */
  typedef FT_Error (*FT_Glyph_InitFunc)( FT_Glyph glyph, FT_GlyphSlot  slot );
  typedef void (*FT_Glyph_DoneFunc)( FT_Glyph glyph );
  typedef void (*FT_Glyph_TransformFunc)( FT_Glyph glyph, const FT_Matrix* matrix, const FT_Vector*  delta );
  typedef void (*FT_Glyph_GetBBoxFunc)( FT_Glyph glyph, FT_BBox* abbox );
  typedef FT_Error (*FT_Glyph_CopyFunc)( FT_Glyph source, FT_Glyph target );
  typedef FT_Error (*FT_Glyph_PrepareFunc)( FT_Glyph glyph, FT_GlyphSlot slot );

  struct  FT_Glyph_Class_
  {
    FT_Long                 glyph_size;
    FT_Glyph_Format         glyph_format;
    FT_Glyph_InitFunc       glyph_init;
    FT_Glyph_DoneFunc       glyph_done;
    FT_Glyph_CopyFunc       glyph_copy;
    FT_Glyph_TransformFunc  glyph_transform;
    FT_Glyph_GetBBoxFunc    glyph_bbox;
    FT_Glyph_PrepareFunc    glyph_prepare;
  };

  typedef FT_Error (*FT_Renderer_RenderFunc)( FT_Renderer renderer, FT_GlyphSlot slot, FT_UInt mode, const FT_Vector*  origin );
  typedef FT_Error (*FT_Renderer_TransformFunc)( FT_Renderer renderer, FT_GlyphSlot slot, const FT_Matrix*  matrix, const FT_Vector*  delta );
  typedef void (*FT_Renderer_GetCBoxFunc)( FT_Renderer renderer, FT_GlyphSlot slot, FT_BBox* cbox );
  typedef FT_Error (*FT_Renderer_SetModeFunc)( FT_Renderer renderer, FT_ULong mode_tag, FT_Pointer mode_ptr );

  typedef struct  FT_Renderer_Class_
  {
    FT_Module_Class            root;

    FT_Glyph_Format            glyph_format;

    FT_Renderer_RenderFunc     render_glyph;
    FT_Renderer_TransformFunc  transform_glyph;
    FT_Renderer_GetCBoxFunc    get_glyph_cbox;
    FT_Renderer_SetModeFunc    set_mode;

    FT_Raster_Funcs*           raster_class;

  } FT_Renderer_Class;

  #define FT_ADVANCE_FLAG_FAST_ONLY  0x20000000UL

  typedef enum  FT_Orientation_
  {
    FT_ORIENTATION_TRUETYPE   = 0,
    FT_ORIENTATION_POSTSCRIPT = 1,
    FT_ORIENTATION_FILL_RIGHT = FT_ORIENTATION_TRUETYPE,
    FT_ORIENTATION_FILL_LEFT  = FT_ORIENTATION_POSTSCRIPT,
    FT_ORIENTATION_NONE

  } FT_Orientation;

  typedef enum  BDF_PropertyType_
  {
    BDF_PROPERTY_TYPE_NONE     = 0,
    BDF_PROPERTY_TYPE_ATOM     = 1,
    BDF_PROPERTY_TYPE_INTEGER  = 2,
    BDF_PROPERTY_TYPE_CARDINAL = 3

  } BDF_PropertyType;

  typedef struct BDF_PropertyRec_*  BDF_Property;
  typedef struct  BDF_PropertyRec_
  {
    BDF_PropertyType  type;
    union {
      const char*     atom;
      FT_Int32        integer;
      FT_UInt32       cardinal;

    } u;

  } BDF_PropertyRec; 

enum {
  FT_Err_Ok = 0x00,
  FT_Err_Cannot_Open_Resource = 0x01,
  FT_Err_Unknown_File_Format = 0x02,
  FT_Err_Invalid_File_Format = 0x03,
  FT_Err_Invalid_Version = 0x04,
  FT_Err_Lower_Module_Version = 0x05,
  FT_Err_Invalid_Argument = 0x06,
  FT_Err_Unimplemented_Feature = 0x07,
  FT_Err_Invalid_Table = 0x08,
  FT_Err_Invalid_Offset = 0x09,
  FT_Err_Array_Too_Large = 0x0A,

  /* glyph/character errors */

  FT_Err_Invalid_Glyph_Index = 0x10,
  FT_Err_Invalid_Character_Code = 0x11,
  FT_Err_Invalid_Glyph_Format = 0x12,
  FT_Err_Cannot_Render_Glyph = 0x13,
  FT_Err_Invalid_Outline = 0x14,
  FT_Err_Invalid_Composite = 0x15,
  FT_Err_Too_Many_Hints = 0x16,
  FT_Err_Invalid_Pixel_Size = 0x17,

  /* handle errors */

  FT_Err_Invalid_Handle = 0x20,
  FT_Err_Invalid_Library_Handle = 0x21,
  FT_Err_Invalid_Driver_Handle = 0x22,
  FT_Err_Invalid_Face_Handle = 0x23,
  FT_Err_Invalid_Size_Handle = 0x24,
  FT_Err_Invalid_Slot_Handle = 0x25,
  FT_Err_Invalid_CharMap_Handle = 0x26,
  FT_Err_Invalid_Cache_Handle = 0x27,
  FT_Err_Invalid_Stream_Handle = 0x28,

  /* driver errors */

  FT_Err_Too_Many_Drivers = 0x30,
  FT_Err_Too_Many_Extensions = 0x31,

  /* memory errors */

  FT_Err_Out_Of_Memory = 0x40,
  FT_Err_Unlisted_Object = 0x41,

  /* stream errors */

  FT_Err_Cannot_Open_Stream = 0x51,
  FT_Err_Invalid_Stream_Seek = 0x52,
  FT_Err_Invalid_Stream_Skip = 0x53,
  FT_Err_Invalid_Stream_Read = 0x54,
  FT_Err_Invalid_Stream_Operation = 0x55,
  FT_Err_Invalid_Frame_Operation = 0x56,
  FT_Err_Nested_Frame_Access = 0x57,
  FT_Err_Invalid_Frame_Read = 0x58,

  /* raster errors */

  FT_Err_Raster_Uninitialized = 0x60,
  FT_Err_Raster_Corrupted = 0x61,
  FT_Err_Raster_Overflow = 0x62,
  FT_Err_Raster_Negative_Height = 0x63,

  /* cache errors */

  FT_Err_Too_Many_Caches = 0x70,

  /* TrueType and SFNT errors */

  FT_Err_Invalid_Opcode = 0x80,
  FT_Err_Too_Few_Arguments = 0x81,
  FT_Err_Stack_Overflow = 0x82,
  FT_Err_Code_Overflow = 0x83,
  FT_Err_Bad_Argument = 0x84,
  FT_Err_Divide_By_Zero = 0x85,
  FT_Err_Invalid_Reference = 0x86,
  FT_Err_Debug_OpCode = 0x87,
  FT_Err_ENDF_In_Exec_Stream = 0x88,
  FT_Err_Nested_DEFS = 0x89,
  FT_Err_Invalid_CodeRange = 0x8A,
  FT_Err_Execution_Too_Long = 0x8B,
  FT_Err_Too_Many_Function_Defs = 0x8C,
  FT_Err_Too_Many_Instruction_Defs = 0x8D,
  FT_Err_Table_Missing = 0x8E,
  FT_Err_Horiz_Header_Missing = 0x8F,
  FT_Err_Locations_Missing = 0x90,
  FT_Err_Name_Table_Missing = 0x91,
  FT_Err_CMap_Table_Missing = 0x92,
  FT_Err_Hmtx_Table_Missing = 0x93,
  FT_Err_Post_Table_Missing = 0x94,
  FT_Err_Invalid_Horiz_Metrics = 0x95,
  FT_Err_Invalid_CharMap_Format = 0x96,
  FT_Err_Invalid_PPem = 0x97,
  FT_Err_Invalid_Vert_Metrics = 0x98,
  FT_Err_Could_Not_Find_Context = 0x99,
  FT_Err_Invalid_Post_Table_Format = 0x9A,
  FT_Err_Invalid_Post_Table = 0x9B,

  /* CFF, CID, and Type 1 errors */
  FT_Err_Syntax_Error = 0xA0,
  FT_Err_Stack_Underflow = 0xA1,
  FT_Err_Ignore = 0xA2,
  FT_Err_No_Unicode_Glyph_Name = 0xA3,

  /* BDF errors */
  FT_Err_Missing_Startfont_Field = 0xB0,
  FT_Err_Missing_Font_Field = 0xB1,
  FT_Err_Missing_Size_Field = 0xB2,
  FT_Err_Missing_Fontboundingbox_Field = 0xB3,
  FT_Err_Missing_Chars_Field = 0xB4,
  FT_Err_Missing_Startchar_Field = 0xB5,
  FT_Err_Missing_Encoding_Field = 0xB6,
  FT_Err_Missing_Bbx_Field = 0xB7,
  FT_Err_Bbx_Too_Big = 0xB8,
  FT_Err_Corrupted_Font_Header = 0xB9,
  FT_Err_Corrupted_Font_Glyphs = 0xBA,
  FT_Err_Max
};

  typedef FT_Fixed  FT_Angle;

#define FT_ANGLE_PI  ( 180L << 16 )
#define FT_ANGLE_2PI  ( FT_ANGLE_PI * 2 )
#define FT_ANGLE_PI2  ( FT_ANGLE_PI / 2 )
#define FT_ANGLE_PI4  ( FT_ANGLE_PI / 4 ) 

#define FT_GASP_NO_TABLE               -1
#define FT_GASP_DO_GRIDFIT           0x01
#define FT_GASP_DO_GRAY              0x02
#define FT_GASP_SYMMETRIC_SMOOTHING  0x08
#define FT_GASP_SYMMETRIC_GRIDFIT    0x10

  typedef struct  FT_SfntName_
  {
    FT_UShort  platform_id;
    FT_UShort  encoding_id;
    FT_UShort  language_id;
    FT_UShort  name_id;

    FT_Byte*   string;      /* this string is *not* null-terminated! */
    FT_UInt    string_len;  /* in bytes */

  } FT_SfntName;

#define FT_PARAM_TAG_IGNORE_PREFERRED_FAMILY  FT_MAKE_TAG( 'i', 'g', 'p', 'f' )
#define FT_PARAM_TAG_IGNORE_PREFERRED_SUBFAMILY  FT_MAKE_TAG( 'i', 'g', 'p', 's' ) 

    typedef struct FT_StrokerRec_*  FT_Stroker;

  typedef enum  FT_Stroker_LineJoin_
  {
    FT_STROKER_LINEJOIN_ROUND = 0,
    FT_STROKER_LINEJOIN_BEVEL,
    FT_STROKER_LINEJOIN_MITER

  } FT_Stroker_LineJoin;

  typedef enum  FT_Stroker_LineCap_
  {
    FT_STROKER_LINECAP_BUTT = 0,
    FT_STROKER_LINECAP_ROUND,
    FT_STROKER_LINECAP_SQUARE

  } FT_Stroker_LineCap;

  typedef enum  FT_StrokerBorder_
  {
    FT_STROKER_BORDER_LEFT = 0,
    FT_STROKER_BORDER_RIGHT

  } FT_StrokerBorder;

NV_FREETYPE_FUN( FT_Error, FT_Init_FreeType, ( FT_Library  *alibrary ) );
NV_FREETYPE_FUN( FT_Error, FT_Done_FreeType, ( FT_Library  library ) );
NV_FREETYPE_FUN( FT_Error, FT_New_Face, ( FT_Library library, const char*  filepathname, FT_Long face_index, FT_Face *aface ) );
NV_FREETYPE_FUN( FT_Error, FT_New_Memory_Face, ( FT_Library library, const FT_Byte*  file_base, FT_Long file_size, FT_Long face_index, FT_Face *aface ) );
NV_FREETYPE_FUN( FT_Error, FT_Open_Face, ( FT_Library library, const FT_Open_Args*  args, FT_Long face_index, FT_Face *aface ) );
NV_FREETYPE_FUN( FT_Error, FT_Attach_File, ( FT_Face face, const char*  filepathname ) );
NV_FREETYPE_FUN( FT_Error, FT_Attach_Stream, ( FT_Face face, FT_Open_Args* parameters ) );
NV_FREETYPE_FUN( FT_Error, FT_Reference_Face, ( FT_Face  face ) );
NV_FREETYPE_FUN( FT_Error, FT_Done_Face, ( FT_Face  face ) );
NV_FREETYPE_FUN( FT_Error, FT_Select_Size, ( FT_Face face, FT_Int strike_index ) );
NV_FREETYPE_FUN( FT_Error, FT_Request_Size, ( FT_Face face, FT_Size_Request req ) );
NV_FREETYPE_FUN( FT_Error, FT_Set_Char_Size, ( FT_Face face, FT_F26Dot6 char_width, FT_F26Dot6 char_height, FT_UInt horz_resolution, FT_UInt vert_resolution ) );
NV_FREETYPE_FUN( FT_Error, FT_Set_Pixel_Sizes, ( FT_Face face, FT_UInt pixel_width, FT_UInt pixel_height ) );
NV_FREETYPE_FUN( FT_Error, FT_Load_Glyph, ( FT_Face face, FT_UInt glyph_index, FT_Int32 load_flags ) );
NV_FREETYPE_FUN( FT_Error, FT_Load_Char, ( FT_Face face, FT_ULong char_code, FT_Int32 load_flags ) );
NV_FREETYPE_FUN( void, FT_Set_Transform, ( FT_Face face, FT_Matrix* matrix, FT_Vector* delta ) );
NV_FREETYPE_FUN( FT_Error, FT_Render_Glyph, ( FT_GlyphSlot slot, FT_Render_Mode render_mode ) );
NV_FREETYPE_FUN( FT_Error, FT_Get_Kerning, ( FT_Face face, FT_UInt left_glyph, FT_UInt right_glyph, FT_UInt kern_mode, FT_Vector *akerning ) );
NV_FREETYPE_FUN( FT_Error, FT_Get_Track_Kerning, ( FT_Face face, FT_Fixed point_size, FT_Int degree, FT_Fixed* akerning ) );
NV_FREETYPE_FUN( FT_Error, FT_Get_Glyph_Name, ( FT_Face face, FT_UInt glyph_index, FT_Pointer buffer, FT_UInt buffer_max ) );
NV_FREETYPE_FUN( const char*, FT_Get_Postscript_Name, ( FT_Face face ) );
NV_FREETYPE_FUN( FT_Error, FT_Select_Charmap, ( FT_Face face, FT_Encoding  encoding ) );
NV_FREETYPE_FUN( FT_Error, FT_Set_Charmap, ( FT_Face face, FT_CharMap charmap ) );
NV_FREETYPE_FUN( FT_Int, FT_Get_Charmap_Index, ( FT_CharMap charmap ) );
NV_FREETYPE_FUN( FT_UInt, FT_Get_Char_Index, ( FT_Face face, FT_ULong charcode ) );
NV_FREETYPE_FUN( FT_ULong, FT_Get_First_Char, ( FT_Face face, FT_UInt *agindex ) );
NV_FREETYPE_FUN( FT_ULong, FT_Get_Next_Char, ( FT_Face face, FT_ULong char_code, FT_UInt *agindex ) );
NV_FREETYPE_FUN( FT_UInt, FT_Get_Name_Index, ( FT_Face face, FT_String* glyph_name ) );
NV_FREETYPE_FUN( FT_Error, FT_Get_SubGlyph_Info, ( FT_GlyphSlot glyph, FT_UInt sub_index, FT_Int *p_index, FT_UInt *p_flags, FT_Int *p_arg1, FT_Int *p_arg2, FT_Matrix *p_transform ) );
NV_FREETYPE_FUN( FT_UShort, FT_Get_FSType_Flags, ( FT_Face  face ) );
NV_FREETYPE_FUN( FT_UInt, FT_Face_GetCharVariantIndex, ( FT_Face face, FT_ULong charcode, FT_ULong variantSelector ) );
NV_FREETYPE_FUN( FT_Int, FT_Face_GetCharVariantIsDefault, ( FT_Face face, FT_ULong charcode, FT_ULong variantSelector ) );
NV_FREETYPE_FUN( FT_UInt32*, FT_Face_GetVariantSelectors, ( FT_Face face ) );
NV_FREETYPE_FUN( FT_UInt32*, FT_Face_GetVariantsOfChar, ( FT_Face face, FT_ULong charcode ) );
NV_FREETYPE_FUN( FT_UInt32*, FT_Face_GetCharsOfVariant, ( FT_Face face, FT_ULong variantSelector ) );
NV_FREETYPE_FUN( FT_Long, FT_MulFix, ( FT_Long  a, FT_Long  b ) );
NV_FREETYPE_FUN( FT_Long, FT_DivFix, ( FT_Long  a, FT_Long  b ) );
NV_FREETYPE_FUN( FT_Fixed, FT_RoundFix, ( FT_Fixed  a ) );
NV_FREETYPE_FUN( FT_Fixed, FT_CeilFix, ( FT_Fixed  a ) );
NV_FREETYPE_FUN( FT_Fixed, FT_FloorFix, ( FT_Fixed  a ) );
NV_FREETYPE_FUN( void, FT_Vector_Transform, ( FT_Vector* vec, const FT_Matrix*  matrix ) );
NV_FREETYPE_FUN( void, FT_Library_Version, ( FT_Library library, FT_Int *amajor, FT_Int *aminor, FT_Int *apatch ) ); 
NV_FREETYPE_FUN( FT_Error, FT_Get_Glyph, ( FT_GlyphSlot slot, FT_Glyph *aglyph ) );
NV_FREETYPE_FUN( FT_Error, FT_Glyph_Copy, ( FT_Glyph source, FT_Glyph *target ) );
NV_FREETYPE_FUN( FT_Error, FT_Glyph_Transform, ( FT_Glyph glyph, FT_Matrix* matrix, FT_Vector* delta ) );
NV_FREETYPE_FUN( void, FT_Glyph_Get_CBox, ( FT_Glyph glyph, FT_UInt bbox_mode, FT_BBox  *acbox ) );
NV_FREETYPE_FUN( FT_Error, FT_Glyph_To_Bitmap, ( FT_Glyph* the_glyph, FT_Render_Mode render_mode, FT_Vector* origin, FT_Bool destroy ) );
NV_FREETYPE_FUN( void, FT_Done_Glyph, ( FT_Glyph  glyph ) );
NV_FREETYPE_FUN( void, FT_Matrix_Multiply, ( const FT_Matrix*  a, FT_Matrix*        b ) );
NV_FREETYPE_FUN( FT_Error, FT_Matrix_Invert, ( FT_Matrix*  matrix ) );
NV_FREETYPE_FUN( FT_Renderer, FT_Get_Renderer, ( FT_Library library, FT_Glyph_Format format ) );
NV_FREETYPE_FUN( FT_Error, FT_Set_Renderer, ( FT_Library library, FT_Renderer renderer, FT_UInt num_params, FT_Parameter* parameters ) );
NV_FREETYPE_FUN( FT_Error, FT_Add_Module, ( FT_Library library, const FT_Module_Class*  clazz ) );
NV_FREETYPE_FUN( FT_Module, FT_Get_Module, ( FT_Library library, const char*  module_name ) );
NV_FREETYPE_FUN( FT_Error, FT_Remove_Module, ( FT_Library  library, FT_Module   module ) );
NV_FREETYPE_FUN( FT_Error, FT_Reference_Library, ( FT_Library  library ) );
NV_FREETYPE_FUN( FT_Error, FT_New_Library, ( FT_Memory memory, FT_Library  *alibrary ) );
NV_FREETYPE_FUN( FT_Error, FT_Done_Library, ( FT_Library  library ) );
NV_FREETYPE_FUN( void, FT_Set_Debug_Hook, ( FT_Library library, FT_UInt hook_index, FT_DebugHook_Func  debug_hook ) );
NV_FREETYPE_FUN( void, FT_Add_Default_Modules, ( FT_Library  library ) );
NV_FREETYPE_FUN( FT_TrueTypeEngineType, FT_Get_TrueType_Engine_Type, ( FT_Library  library ) );
NV_FREETYPE_FUN( FT_Error, FT_Get_Advance, ( FT_Face face, FT_UInt gindex, FT_Int32 load_flags, FT_Fixed *padvance ) );
NV_FREETYPE_FUN( FT_Error, FT_Get_Advances, ( FT_Face face, FT_UInt start, FT_UInt count, FT_Int32 load_flags, FT_Fixed *padvances ) ); 
NV_FREETYPE_FUN( FT_Error, FT_Outline_Decompose, ( FT_Outline* outline, const FT_Outline_Funcs*  func_interface, void* user ) );
NV_FREETYPE_FUN( FT_Error, FT_Outline_New, ( FT_Library library, FT_UInt numPoints, FT_Int numContours, FT_Outline  *anoutline ) );
NV_FREETYPE_FUN( FT_Error, FT_Outline_New_Internal, ( FT_Memory memory, FT_UInt numPoints, FT_Int numContours, FT_Outline  *anoutline ) );
NV_FREETYPE_FUN( FT_Error, FT_Outline_Done, ( FT_Library library, FT_Outline*  outline ) );
NV_FREETYPE_FUN( FT_Error, FT_Outline_Done_Internal, ( FT_Memory memory, FT_Outline* outline ) );
NV_FREETYPE_FUN( FT_Error, FT_Outline_Check, ( FT_Outline*  outline ) );
NV_FREETYPE_FUN( void, FT_Outline_Get_CBox, ( const FT_Outline* outline, FT_BBox *acbox ) );
NV_FREETYPE_FUN( void, FT_Outline_Translate, ( const FT_Outline* outline, FT_Pos xOffset, FT_Pos yOffset ) );
NV_FREETYPE_FUN( FT_Error, FT_Outline_Copy, ( const FT_Outline* source, FT_Outline *target ) );
NV_FREETYPE_FUN( void, FT_Outline_Transform, ( const FT_Outline* outline, const FT_Matrix* matrix ) );
NV_FREETYPE_FUN( FT_Error, FT_Outline_Embolden, ( FT_Outline* outline, FT_Pos strength ) );
NV_FREETYPE_FUN( void, FT_Outline_Reverse, ( FT_Outline*  outline ) );
NV_FREETYPE_FUN( FT_Error, FT_Outline_Get_Bitmap, ( FT_Library library, FT_Outline* outline, const FT_Bitmap *abitmap ) );
NV_FREETYPE_FUN( FT_Error, FT_Outline_Render, ( FT_Library library, FT_Outline* outline, FT_Raster_Params* params ) );
NV_FREETYPE_FUN( FT_Orientation, FT_Outline_Get_Orientation, ( FT_Outline*  outline ) );
NV_FREETYPE_FUN( FT_Error, FT_Outline_Get_BBox, ( FT_Outline* outline, FT_BBox *abbox ) );
NV_FREETYPE_FUN( void, FT_Bitmap_New, ( FT_Bitmap  *abitmap ) );
NV_FREETYPE_FUN( FT_Error, FT_Bitmap_Copy, ( FT_Library library, const FT_Bitmap *source, FT_Bitmap *target ) );
NV_FREETYPE_FUN( FT_Error, FT_Bitmap_Embolden, ( FT_Library library, FT_Bitmap* bitmap, FT_Pos xStrength, FT_Pos yStrength ) );
NV_FREETYPE_FUN( FT_Error, FT_Bitmap_Convert, ( FT_Library library, const FT_Bitmap *source, FT_Bitmap *target, FT_Int alignment ) );
NV_FREETYPE_FUN( FT_Error, FT_GlyphSlot_Own_Bitmap, ( FT_GlyphSlot  slot ) );
NV_FREETYPE_FUN( FT_Error, FT_Bitmap_Done, ( FT_Library library, FT_Bitmap *bitmap ) );
NV_FREETYPE_FUN( FT_Error, FT_Get_BDF_Charset_ID, ( FT_Face face, const char* *acharset_encoding, const char* *acharset_registry ) );
NV_FREETYPE_FUN( FT_Error, FT_Get_BDF_Property, ( FT_Face face, const char* prop_name, BDF_PropertyRec *aproperty ) );
NV_FREETYPE_FUN( FT_Error, FT_Get_CID_Registry_Ordering_Supplement, ( FT_Face face, const char* *registry, const char* *ordering, FT_Int *supplement ) );
NV_FREETYPE_FUN( FT_Error, FT_Get_CID_Is_Internally_CID_Keyed, ( FT_Face face, FT_Bool *is_cid ) );
NV_FREETYPE_FUN( FT_Error, FT_Get_CID_From_Glyph_Index, ( FT_Face face, FT_UInt glyph_index, FT_UInt *cid ) );
NV_FREETYPE_FUN( FT_Error, FT_Get_PFR_Metrics, ( FT_Face face, FT_UInt *aoutline_resolution, FT_UInt   *ametrics_resolution, FT_Fixed  *ametrics_x_scale, FT_Fixed  *ametrics_y_scale ) );
NV_FREETYPE_FUN( FT_Error, FT_Get_PFR_Kerning, ( FT_Face face, FT_UInt left, FT_UInt right, FT_Vector  *avector ) );
NV_FREETYPE_FUN( FT_Error, FT_Get_PFR_Advance, ( FT_Face face, FT_UInt gindex, FT_Pos *aadvance ) );
NV_FREETYPE_FUN( FT_Int, FT_Get_Gasp, ( FT_Face face, FT_UInt ppem ) ); 
NV_FREETYPE_FUN( FT_Error, FT_New_Size, ( FT_Face face, FT_Size* size ) );
NV_FREETYPE_FUN( FT_Error, FT_Done_Size, ( FT_Size size ) );
NV_FREETYPE_FUN( FT_Error, FT_Activate_Size, ( FT_Size  size ) );
NV_FREETYPE_FUN( FT_UInt, FT_Get_Sfnt_Name_Count, ( FT_Face  face ) ); 
NV_FREETYPE_FUN( FT_Error, FT_Get_Sfnt_Name, ( FT_Face face, FT_UInt idx, FT_SfntName  *aname ) );

NV_FREETYPE_FUN( FT_StrokerBorder, FT_Outline_GetInsideBorder, ( FT_Outline* outline ) );
NV_FREETYPE_FUN( FT_StrokerBorder, FT_Outline_GetOutsideBorder, ( FT_Outline* outline ) );
NV_FREETYPE_FUN( FT_Error, FT_Stroker_New, ( FT_Library library, FT_Stroker *astroker ) );
NV_FREETYPE_FUN( void, FT_Stroker_Set, ( FT_Stroker stroker, FT_Fixed radius, FT_Stroker_LineCap line_cap, FT_Stroker_LineJoin line_join, FT_Fixed miter_limit ) );
NV_FREETYPE_FUN( void, FT_Stroker_Rewind, ( FT_Stroker stroker ) );
NV_FREETYPE_FUN( FT_Error, FT_Stroker_ParseOutline, ( FT_Stroker stroker, FT_Outline* outline, FT_Bool opened ) );
NV_FREETYPE_FUN( FT_Error, FT_Stroker_BeginSubPath, ( FT_Stroker stroker, FT_Vector* to, FT_Bool open ) );
NV_FREETYPE_FUN( FT_Error, FT_Stroker_EndSubPath, ( FT_Stroker stroker ) );
NV_FREETYPE_FUN( FT_Error, FT_Stroker_LineTo, ( FT_Stroker stroker, FT_Vector* to ) );
NV_FREETYPE_FUN( FT_Error, FT_Stroker_ConicTo, ( FT_Stroker stroker, FT_Vector* control, FT_Vector* to ) );
NV_FREETYPE_FUN( FT_Error, FT_Stroker_CubicTo, ( FT_Stroker stroker, FT_Vector* control1, FT_Vector* control2, FT_Vector* to ) );
NV_FREETYPE_FUN( FT_Error, FT_Stroker_GetBorderCounts, ( FT_Stroker stroker, FT_StrokerBorder border, FT_UInt *anum_points, FT_UInt *anum_contours ) );
NV_FREETYPE_FUN( void, FT_Stroker_ExportBorder, ( FT_Stroker stroker, FT_StrokerBorder border, FT_Outline* outline ) );
NV_FREETYPE_FUN( FT_Error, FT_Stroker_GetCounts, ( FT_Stroker stroker, FT_UInt *anum_points, FT_UInt *anum_contours ) );
NV_FREETYPE_FUN( void, FT_Stroker_Export, ( FT_Stroker stroker, FT_Outline* outline ) );
NV_FREETYPE_FUN( void, FT_Stroker_Done, ( FT_Stroker stroker ) );
NV_FREETYPE_FUN( FT_Error, FT_Glyph_Stroke, ( FT_Glyph *pglyph, FT_Stroker stroker, FT_Bool destroy ) );
NV_FREETYPE_FUN( FT_Error, FT_Glyph_StrokeBorder, ( FT_Glyph *pglyph, FT_Stroker stroker, FT_Bool inside, FT_Bool destroy ) );

NV_FREETYPE_FUN( FT_Fixed, FT_Sin, ( FT_Angle angle ) );
NV_FREETYPE_FUN( FT_Fixed, FT_Cos, ( FT_Angle angle ) );
NV_FREETYPE_FUN( FT_Fixed, FT_Tan, ( FT_Angle angle ) );
NV_FREETYPE_FUN( FT_Angle, FT_Atan2, ( FT_Fixed x, FT_Fixed  y ) );
NV_FREETYPE_FUN( FT_Angle, FT_Angle_Diff, ( FT_Angle angle1, FT_Angle  angle2 ) );
NV_FREETYPE_FUN( void, FT_Vector_Unit, ( FT_Vector* vec, FT_Angle angle ) );
NV_FREETYPE_FUN( void, FT_Vector_Rotate, ( FT_Vector* vec, FT_Angle angle ) );
NV_FREETYPE_FUN( FT_Fixed, FT_Vector_Length, ( FT_Vector* vec ) );
NV_FREETYPE_FUN( void, FT_Vector_Polarize, ( FT_Vector* vec, FT_Fixed *length, FT_Angle *angle ) );
NV_FREETYPE_FUN( void, FT_Vector_From_Polar, ( FT_Vector* vec, FT_Fixed length, FT_Angle angle ) );
NV_FREETYPE_FUN( FT_Error, FT_Library_SetLcdFilter, ( FT_Library library, FT_LcdFilter  filter ) );
NV_FREETYPE_FUN( FT_Error, FT_Library_SetLcdFilterWeights, ( FT_Library library, unsigned char *weights ) );
}

namespace nv {
/* Dynamic load support */
#	if defined( NV_FREETYPE_DYNAMIC )
		bool load_freetype_library( const char* path = NV_FREETYPE_PATH );
#	else
		inline bool load_freetype_library( const char* path = "" ) { return true; }
#	endif
}

#endif // NV_LIB_FREETYPE_HH
