// Copyright (C) 2012-2013 ChaosForge / Kornel Kisielewicz
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

/**
 * @file gui_element.hh
 * @author Kornel Kisielewicz
 * @brief GUI Element
 */

#ifndef NV_GUI_ELEMENT_HH
#define NV_GUI_ELEMENT_HH

#include <nv/object.hh>
#include <nv/position.hh>
#include <nv/io_event.hh>
#include <nv/gui/gui_common.hh>

namespace nv
{
	namespace gui
	{
		class element : public object
		{
		public:

			/**
			 *Creates a new GUI element.
			 */
			element() : object() {}

			/**
			 *Creates a new GUI element with the give root node and postioning data.
			 *
			 *@param aroot The root object that will be this object's parent.
			 *@param r The rectangle representing the position and size of this element.
			 */
			element( root* aroot, const rectangle& r );

			/**
			 *Gets the deepest child element that contains the indicated point.
			 *
			 *@param p The position to check.
			 *@returns The last child element that contains the given point.
			 */
			element* get_element( const position& p );

			/**
			 *Event handler for update events.  Calls on_update for all affected children.
			 *
			 *@param elapsed Time since last update.
			 */
			virtual void on_update( uint32 elapsed );

			/**
			 *Event handler for draw events.  Calls on_draw for all affected children.
			 */
			virtual void on_draw();

			/**
			 *Event handler for IO events.  Class on_event for all affected children.
			 *
			 *@param event The event data.
			 *@returns ???
			 */
			virtual bool on_event( const io_event& event );

			/**
			 *Checks if this element contains the given point.
			 *
			 *@param p The point to check.
			 *@returns True if the point is within the region of the element, false otherwise.
			 */
			virtual bool contains( const position& p ) const;

			/**
			 *Sets the position and size of this element relative to its parent.
			 *
			 *@param r The new position and size of the element.
			 */
			virtual void set_relative( const rectangle& r );

			/**
			 *Sets the position of this element relative to its parent.
			 *
			 *@param p The new position of the element.
			 */
			virtual void set_relative( const position& p );

			/**
			 *Gets the position and size of the element relative to its parent.
			 *
			 *@returns The element's position and size relative to its parent.
			 */
			virtual const rectangle& get_relative() const { return m_relative; }

			/**
			 *Gets the position and size of the element relative to the root (window).
			 *
			 *@returns The element's position and size relative to the root (window).
			 */
			virtual const rectangle& get_absolute() const { return m_absolute; }

			/**
			 *Gets the parent element of this element.
			 *
			 *@returns The parent element.
			 */
			virtual element* get_parent() const { return (element*)m_parent; }

			/**
			 *Gets whether this element is currently accepting events.
			 *
			 *@returns True if the element is receiving events, false otherwise.
			 */
			virtual bool is_enabled() const { return m_enabled; }

			/**
			 *Gets whether this element is currently visible.
			 *
			 *@returns True if the element is visible, false otherwise.
			 */
			virtual bool is_visible() const { return m_visible; }

			/**
			 *Gets whether this element needs to be redrawn.
			 *
			 *@returns True if the element needs to be redrawn, false if not.
			 */
			virtual bool is_dirty()   const { return m_dirty; }

			/**
			 *Sets whether this element is currently accepting events.
			 *
			 *@param value True to allow the element and its children to receive events, false to disable.
			 */
			virtual void set_enabled( bool value ) { m_enabled = value; }

			/**
			 *Sets whether this element is visible on the screen.
			 *
			 *@param value True to display the element and its children, false to hide it and its children.
			 */
			virtual void set_visible( bool value ) { m_visible = value; }

			/**
			 *Sets whether this element needs to be redrawn.
			 *
			 *@param value True to request that the element and its children be redrawn, false if not.
			 */
			virtual void set_dirty( bool value )   { m_dirty   = value; }

			/**
			 *Gets the text associated with this element.
			 *
			 *@returns A string containing the associated text.
			 */
			virtual const string& get_text() const { return m_text; }

			/**
			 *Sets the text associated with this element.
			 *
			 *@param text The new text to associate with this element.
			 */
			virtual void set_text( const string& text ) { m_text = text; m_dirty = true; }

			/**
			 *Gets the class name associated with this element.
			 *
			 *@returns A string containing the class name of this element.
			 */
			virtual const string& get_class() const { return m_class; }

			/**
			 *sets the class name associated with this element.
			 *
			 *@param class_ The new class name.
			 */
			virtual void set_class( const string& class_ ) { m_class = class_; m_dirty = true; }

			/**
			 *Recalcuates the absolute position of the element and the element's children.
			 */
			virtual void recalculate_absolute();

			/**
			 *Recalculates the aboslute position of the element's children.
			 */
			virtual void recalculate_absolute_children();

			/**
			 *Destroys the element.
			 */
			virtual ~element();
		protected:
			friend class environment;

			string    m_class; ///< Class name.
			string    m_text; ///< Displayed label or text.
			rectangle m_relative; ///< Position relative to parent.
			rectangle m_absolute; ///< Position relative to window/screen.
			bool m_enabled; ///< Whether the element accepts events.
			bool m_visible; ///< Whether the element is drawn.
			bool m_dirty; ///< Whether the element needs updating.
			render_data* m_render_data; ///<   -?-
		};

	} // namespace gui

} // namespace nv

#endif // NV_GUI_ELEMENT_HH
