// Copyright (C) 2012 Kornel Kisielewicz
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

#ifndef NV_TYPES_HH
#define NV_TYPES_HH

#include <glm/glm.hpp>
#include <nv/common.hh>
#include <nv/object.hh>
#include <type_traits>
#include <typeinfo>
#include <typeindex>
#include <utility>
#include <unordered_map>
#include <vector>
#include <string>
#include <cstddef>

namespace nv
{
	enum datatype
	{
		INT,
		BYTE,
		SHORT,
		UINT,
		UBYTE,
		USHORT,
		FLOAT,
		FLOAT_VECTOR_2,
		FLOAT_VECTOR_3,
		FLOAT_VECTOR_4,
		FLOAT_MATRIX_2,
		FLOAT_MATRIX_3,
		FLOAT_MATRIX_4,
		INT_VECTOR_2,
		INT_VECTOR_3,
		INT_VECTOR_4,
		// unsupported gl conversion, remove?
		BYTE_VECTOR_2, 
		BYTE_VECTOR_3,
		BYTE_VECTOR_4,
	};

	template <typename T> 
	struct datatype_traits 
	{
		typedef T type;
		typedef T base_type;
		static const size_t size = 1;
	};

	template <typename T> 
	struct datatype_traits< glm::detail::tvec2<T> > 
	{
		typedef glm::detail::tvec2<T> type;
		typedef typename type::value_type base_type;
		static const size_t size = 2;
	};

	template <typename T> 
	struct datatype_traits< glm::detail::tvec3<T> > 
	{
		typedef glm::detail::tvec3<T> type;
		typedef typename type::value_type base_type;
		static const size_t size = 3;
	};

	template <typename T> 
	struct datatype_traits< glm::detail::tvec4<T> > 
	{
		typedef glm::detail::tvec4<T> type;
		typedef typename type::value_type base_type;
		static const size_t size = 4;
	};

	typedef glm::detail::tvec2<sint8> i8vec2;
	typedef glm::detail::tvec3<sint8> i8vec3;
	typedef glm::detail::tvec4<sint8> i8vec4;

	typedef glm::vec2 vec2;
	typedef glm::vec3 vec3;
	typedef glm::vec4 vec4;

	typedef glm::ivec2 ivec2;
	typedef glm::ivec3 ivec3;
	typedef glm::ivec4 ivec4;

	typedef glm::mat2 mat2;
	typedef glm::mat3 mat3;
	typedef glm::mat4 mat4;

	template < datatype EnumType > struct enum_to_type {};

	template <> struct enum_to_type< INT >   { typedef int type; };
	template <> struct enum_to_type< UINT >  { typedef unsigned int type; };
	template <> struct enum_to_type< SHORT > { typedef short type; };
	template <> struct enum_to_type< USHORT >{ typedef unsigned short type; };
	template <> struct enum_to_type< BYTE >  { typedef char type; };
	template <> struct enum_to_type< UBYTE > { typedef unsigned char type; };
	template <> struct enum_to_type< FLOAT > { typedef f32 type; };

	template <> struct enum_to_type< FLOAT_VECTOR_2 > { typedef vec2 type; };
	template <> struct enum_to_type< FLOAT_VECTOR_3 > { typedef vec3 type; };
	template <> struct enum_to_type< FLOAT_VECTOR_4 > { typedef vec4 type; };

	template <> struct enum_to_type< INT_VECTOR_2 > { typedef ivec2 type; };
	template <> struct enum_to_type< INT_VECTOR_3 > { typedef ivec3 type; };
	template <> struct enum_to_type< INT_VECTOR_4 > { typedef ivec4 type; };

	template <> struct enum_to_type< BYTE_VECTOR_2 > { typedef i8vec2 type; };
	template <> struct enum_to_type< BYTE_VECTOR_3 > { typedef i8vec3 type; };
	template <> struct enum_to_type< BYTE_VECTOR_4 > { typedef i8vec4 type; };

	template <> struct enum_to_type< FLOAT_MATRIX_2 > { typedef mat2 type; };
	template <> struct enum_to_type< FLOAT_MATRIX_3 > { typedef mat3 type; };
	template <> struct enum_to_type< FLOAT_MATRIX_4 > { typedef mat4 type; };

	template < typename TYPE > struct type_to_enum {};

	template <> struct type_to_enum< int >           { static const datatype type = INT; };
	template <> struct type_to_enum< unsigned int >  { static const datatype type = UINT; };
	template <> struct type_to_enum< short >         { static const datatype type = SHORT; };
	template <> struct type_to_enum< unsigned short >{ static const datatype type = USHORT; };
	template <> struct type_to_enum< char >          { static const datatype type = BYTE; };
	template <> struct type_to_enum< signed char >   { static const datatype type = BYTE; };
	template <> struct type_to_enum< unsigned char > { static const datatype type = UBYTE; };
	template <> struct type_to_enum< f32 > { static const datatype type = FLOAT; };

	template <> struct type_to_enum< vec2 > { static const datatype type = FLOAT_VECTOR_2; };
	template <> struct type_to_enum< vec3 > { static const datatype type = FLOAT_VECTOR_3; };
	template <> struct type_to_enum< vec4 > { static const datatype type = FLOAT_VECTOR_4; };

	template <> struct type_to_enum< ivec2 > { static const datatype type = INT_VECTOR_2; };
	template <> struct type_to_enum< ivec3 > { static const datatype type = INT_VECTOR_3; };
	template <> struct type_to_enum< ivec4 > { static const datatype type = INT_VECTOR_4; };

	template <> struct type_to_enum< i8vec2 > { static const datatype type = BYTE_VECTOR_2; };
	template <> struct type_to_enum< i8vec3 > { static const datatype type = BYTE_VECTOR_3; };
	template <> struct type_to_enum< i8vec4 > { static const datatype type = BYTE_VECTOR_4; };

	template <> struct type_to_enum< mat2 > { static const datatype type = FLOAT_MATRIX_2; };
	template <> struct type_to_enum< mat3 > { static const datatype type = FLOAT_MATRIX_3; };
	template <> struct type_to_enum< mat4 > { static const datatype type = FLOAT_MATRIX_4; };

	template<typename T>
	struct is_container
	{
		private:
			typedef char                      yes;
			typedef struct { char array[2]; } no;
			template<typename C> static yes test(typename C::iterator*);
			template<typename C> static no  test(...);
		public:
			static const bool value = sizeof(test<T>(0)) == sizeof(yes);
	};

	template<>
	struct is_container< std::string > {
		static const bool value = false;
	};

	template <typename TYPE> void ConstructObject(void* object)
	{
		// Use placement new to call the constructor
		new (object) TYPE;
	}
	template <typename TYPE> void DestructObject(void* object)
	{
		// Explicit call of the destructor
		((TYPE*)object)->TYPE::~TYPE();
	}
    
	struct type_entry;

	enum type_flag
	{
		TF_POINTER      = 0x01, //< field is a pointer
		TF_NOSERIALIZE  = 0x02, //< ignore during serialization
		TF_INVISIBLE    = 0x04, //< field invisible to API
		TF_READONLY     = 0x08, //< read only field
		TF_SIMPLETYPE   = 0x10, //< raw binary I/O possible
		TF_OWNED        = 0x20,
		TF_CONTAINER    = 0x40, //< is a container
	};

	struct type_field
	{
		std::string      name;      //< name of the field
		std::string      type_name; //< name of the type of the field
		const std::type_info* type_inf;  //< typeinfo for later retrieval of type
		type_entry*      type;      //< pointer to field type
		unsigned int     flags;     //< flags 
		size_t           offset;

		template< typename TOBJECT, typename TFIELD >
		type_field( const char* name, TFIELD TOBJECT::*field, typename std::enable_if< is_container<TFIELD>::value, void* >::type = nullptr )
		: name(name)
			, type_name()
			, type_inf( &typeid( typename std::remove_pointer<typename TFIELD::value_type>::type ) )
			, type( nullptr )
			, flags( 0 )
			, offset( offset_of( field ) ) 
		{
			NV_LOG( LOG_INFO, name << "-" << offset);
			flags = TF_CONTAINER |
				( std::is_pointer<typename TFIELD::value_type>::value ? TF_POINTER : 0 ) |
				( std::is_pod<typename TFIELD::value_type>::value ? TF_SIMPLETYPE : 0 );
		}

		template< typename TOBJECT, typename TFIELD >
		type_field( const char* name, TFIELD TOBJECT::*field, typename std::enable_if< !is_container<TFIELD>::value, void* >::type = nullptr )
		: name(name)
			, type_name()
			, type_inf( &typeid( typename std::remove_pointer<TFIELD>::type ) )
			, type( nullptr )
			, flags( 0 )
			, offset( offset_of( field ) )
		{
			NV_LOG( LOG_INFO, name << "-" << offset);
			flags = 
				( std::is_pointer<TFIELD>::value ? TF_POINTER : 0 ) |
				( std::is_pod<TFIELD>::value ? TF_SIMPLETYPE : 0 );
		}

		type_field& flag( unsigned int f )
		{
			flags |= f;
			return *this;
		}
	};

	struct type_enum
	{
		std::string name;
		int         value;
		type_enum( const char* name, int value ) : name(name), value(value) {}
	};

	struct type_entry
	{
		// Function types for the constructor and destructor of registered types
		typedef void (*constructor_func)(void*);
		typedef void (*destructor_func)(void*);

		// Parent type database
		type_database* type_db;

		// Scoped C++ name of the type
		std::string name;

		// Pointers to the constructor and destructor functions
		constructor_func constructor;
		destructor_func  destructor;

		// Result of sizeof(type) operation
		size_t size;

		// Base type
		type_entry* base_type;

		// Field list
		std::vector<type_field> field_list;

		// Enum list
		std::vector<type_enum> enum_list;

		template <typename TYPE>
		type_entry& base();

		template <int SIZE>
		type_entry& fields( type_field (&init_fields)[SIZE] );

		template <int SIZE>
		type_entry& enums( type_enum (&init_enums)[SIZE] )
		{
			for (int i = 0; i < SIZE; i++)
			{
				enum_list.push_back( init_enums[i] );
			}
			return *this;
		}
	};

	class type_database
	{
	public:
		template< typename TYPE >
		type_entry& create_type( const char* name )
		{
			type_entry* i_type = nullptr;
			type_name_map::iterator it = m_name_types.find( name );
			if ( it != m_name_types.end() ) 
			{
				return *(it->second);
			}
			i_type          = new type_entry;
			i_type->type_db = this;
			i_type->name    = name;
			i_type->size    = sizeof(TYPE);

			i_type->constructor = ConstructObject<TYPE>;
			i_type->destructor  = DestructObject<TYPE>;

			m_name_types[name]        = i_type;
			m_idx_types[typeid(TYPE)] = i_type;
			return *i_type;
		}

		type_entry* get_type( const std::string name )
		{
			type_name_map::iterator it = m_name_types.find( name );
			if ( it != m_name_types.end() ) 
			{
				return it->second;
			}
			return nullptr;
		}

		type_entry* get_type( const std::type_info& t )
		{
			type_info_map::iterator it = m_idx_types.find( std::type_index(t) );
			if ( it != m_idx_types.end() ) 
			{
				return it->second;
			}
			return nullptr;
		}
	private:
		struct compare_type_info {
			bool operator ()(const std::type_info* a, const std::type_info* b) const {
				return a->before(*b);
			}
		};

		typedef std::unordered_map<std::string, type_entry*>     type_name_map;
		typedef std::unordered_map<std::type_index, type_entry*> type_info_map;
		type_name_map m_name_types;
		type_info_map m_idx_types;
	};
    
    template < typename TYPE >
    type_entry& type_entry::base()
    {
        base_type = type_db->get_type( typeid(TYPE) );
    }
    
    template <int SIZE>
    type_entry& type_entry::fields( type_field (&init_fields)[SIZE] )
    {
        for (int i = 0; i < SIZE; i++)
        {
            type_field f = init_fields[i];
            f.type = type_db->get_type(*(f.type_inf));
            f.type_name = f.type->name;
            field_list.push_back(f);
        }
        return *this;
    }

}

#endif

