// Copyright (C) 2012-2013 ChaosForge / Kornel Kisielewicz
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

// TODO : Force no resize on the buffer_slice!
// TODO : Multiple update support?

#ifndef NV_CACHED_BUFFER_HH
#define NV_CACHED_BUFFER_HH

#include <nv/common.hh>
#include <nv/interface/vertex_buffer.hh>

namespace nv
{
	template< typename T > class cached_buffer;

	template< typename T >
	class buffer_slice
	{
	public:
		typedef cached_buffer<T> cache;
		typedef std::vector<T>  vector;

		buffer_slice( cache* c ) 
			: m_cache( c ), m_offset( 0 ), m_cached_size( 0 ), m_locked( true )
		{
			m_cache->reset();
		}
		void commit()
		{
			m_cache->commit( this );
			m_cached_size = m_data.size();
			m_locked      = false;
		}
		vector& lock()
		{
			m_locked = true;
			return m_data;
		}
		const vector& data() 
		{ 
			return m_data; 
		}
		~buffer_slice()
		{
			m_cache->reset();
		}
	private:
		friend class cached_buffer<T>;

		vector m_data;
		cache* m_cache;
		size_t m_offset;
		size_t m_cached_size;
		bool   m_locked;
	};

	template< typename T >
	class cached_buffer
	{
	public:
		typedef buffer_slice<T> slice;
		typedef std::vector<T>  vector;
		typedef T               value_type;
		static const int value_type_size = sizeof(T);

		cached_buffer( device* dev, buffer_hint hint, int initial_size, bool is_vertex = true ) 
			: m_device( dev )
			, m_buffer( nullptr )
			, m_hint( hint )
			, m_full_update( true )
			, m_is_vertex( is_vertex )
			, m_data()
			, m_min(0)
			, m_max(0)
		{ 
			create_buffer( initial_size );
		}

		void commit( slice* bslice )
		{
			if ( m_full_update )
			{
				const vector& bv = bslice->data();
				bslice->m_offset = m_data.size();
				m_data.insert( m_data.end(), bv.cbegin(), bv.cend() );
			}
			else if ( bslice->m_locked )
			{
				const vector& bv = bslice->data();
				if ( bslice->m_cached_size != bv.size() ) 
				{
					m_data.erase( m_data.begin() + bslice->m_offset, m_data.end() );
					m_full_update = true;
					m_data.insert( m_data.end(), bv.cbegin(), bv.cend() );
					m_min = glm::min<int>( m_min, bslice->m_offset );
				}
				else
				{
					std::copy( bv.cbegin(), bv.cend(), m_data.begin() + bslice->m_offset );
					m_min = glm::min<int>( m_min, bslice->m_offset );
					m_max = glm::max<int>( m_max, bslice->m_offset + bv.size() );
				}
			}
		}

		void reset()
		{
			m_data.clear();
			m_full_update = true;
			m_min = 0;
			m_max = 0;
		}

		/**
		 * Returns true if buffer has been recreated
		 */
		bool commit()
		{
			bool result = false;
			size_t bsize = (size_t)get_max_size();
			if ( m_data.size() > bsize )
			{
				while( m_data.size() > bsize ) bsize *= 2;
				create_buffer( bsize );
				m_full_update = true;
				result = true;
			}
			m_buffer->bind();
			if ( m_full_update )
			{
				if ( m_min > 0 )
				{
					int offset = m_min * value_type_size;
					int size   = m_data.size() * value_type_size - offset;
					m_buffer->update( m_data.data() + m_min, offset, size );
				}
				else
				{
					m_buffer->update( m_data.data(), 0, m_data.size() * value_type_size );
				}
			}
			else if ( m_max > 0 )
			{
				int offset = m_min * value_type_size;
				int size   = (m_max-m_min) * value_type_size;
				m_buffer->update( m_data.data() + m_min, offset, size );
			}
			m_buffer->unbind();
			m_full_update = false;
			m_min = get_max_size();
			m_max = 0;
			return result;
		}

		int get_max_size() const
		{ 
			return m_buffer->get_size() / value_type_size;
		}

		int get_size() const
		{ 
			return m_data.size(); 
		}

		buffer* get_buffer()
		{ 
			return m_buffer; 
		}

		virtual ~cached_buffer() 
		{
			delete m_buffer;
		}
	private:
		void create_buffer( int size )
		{
			delete m_buffer;
			if ( m_is_vertex )
				m_buffer = m_device->create_vertex_buffer( m_hint, size * value_type_size, nullptr );
			else
				m_buffer = m_device->create_index_buffer( m_hint, size * value_type_size, nullptr );
		}
	private:
		device*     m_device;
		buffer*     m_buffer;
		buffer_hint m_hint;
		bool        m_full_update;
		bool        m_is_vertex;
		vector      m_data;

		int         m_min;
		int         m_max;
	};

} // namespace nv

#endif // NV_CACHED_BUFFER_HH
