// Copyright (C) 2012-2013 ChaosForge / Kornel Kisielewicz
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

/**
 * @file gui_element.hh
 * @author Kornel Kisielewicz
 * @brief UI 2D positioning classes
 */

#ifndef NV_POSITION_HH
#define NV_POSITION_HH

#include <glm/glm.hpp>
#include <nv/common.hh>
#include <utility>

namespace nv
{
	typedef glm::ivec2 position;
	typedef glm::ivec2 dimension;

	struct rectangle
	{
		typedef position::value_type value_type;
		typedef std::size_t size_type;
		typedef rectangle type;
		
		union 
		{
			struct { position upper_left; position lower_right; };
			struct { position ul;         position lr; };
			struct { value_type x1,y1,x2,y2; };
		};
		/**
		 *Creates a new rectangle assigned to {0, 0, 0, 0}.
		 */
		rectangle() : ul(), lr() {}
		
		/**
		 *Creates a new rectangle given a position.
		 *
		 *@param p The position to assign the rectangle to.
		 */
		rectangle( position p ) : ul(p), lr(p) {}
		
		/**
		 *Creates a new rectangle given an upper-left and lower-right position.
		 *
		 *@param ul The position of the upper-left corner of the rectangle.
		 *@param lr The position of the lower-right corner of the rectangle.
		 */
		rectangle( position ul, position lr ) : ul(ul), lr(lr) {}
		
		/**
		 *Creates a new rectangle given an upper-left position, width, and height.
		 *
		 *@param ul The position of the upper-left corner of the rectangle.
		 *@param width The width of the rectangle.
		 *@param height The height of the rectangle.
		 */
		rectangle( position ul, value_type width, value_type height ) : ul(ul), lr(ul + position(width,height)) {}
		
		/**
		 *Sets the dimensions of the rectangle without moving the upper-left of the rectangle.
		 *
		 *@param d The new dimensions of the rectangle.
		 */
		rectangle& dim( dimension d ) { lr = ul + d; return *this; }

		/**
		 *Moves the rectangle to a new position while maintaining its size.
		 *
		 *@param p The new position of the rectangle's upper-left corner.
		 */
		rectangle& pos( position p )  { lr = p + (lr - ul); lr = p; return *this; }

		/**
		 *Sets the dimensions of the rectangle without moving the upper-left of the rectangle.
		 *
		 *@param d The new dimensions of the rectangle.
		 */
		void set_dimension( dimension d ) { lr = ul + d; }
		
		/**
		 *Moves the rectangle to a new position while maintaining its size.
		 *
		 *@param p The new position of the rectangle's upper-left corner.
		 */
		void set_position( position p ) { lr = p + (lr - ul); ul = p; }

		/**
		 *Gets the dimensions of the rectangle.  Synonym for get_size.
		 *
		 *@returns The dimensions of the rectangle.
		 *@see get_size
		 */
		dimension get_dimension() const { return lr - ul; }

		/**
		 *Gets the position of the upper-left corner of the rectangle.
		 *
		 *@returns The position of the rectangle's upper-left corner.
		 */
		position  get_position() const { return ul; }

		/**
		 *Gets the dimensions of the rectangle.  Synonym for get_dimension.
		 *
		 *@returns The dimensions of the rectangle.
		 *@see get_dimension
		 */
		dimension get_size() const { return lr - ul; }

		/**
		 *Gets the center of the rectangle.
		 *
		 *@returns The center of the rectangle.
		 */
		position get_center() const { return ( lr + ul ) / 2; }

		/**
		 *Gets the width of the rectangle.
		 *
		 *@returns The width of the rectangle.
		 */
		value_type get_width() const { return x2 - x1; }

		/**
		 *Gets the height of the rectangle.
		 *
		 *@returns The height of the rectangle.
		 */
		value_type get_height() const { return y2 - y1; }

		/**
		 *Gets the area of the rectangle.
		 *
		 *@returns The area of the rectangle.
		 */
		value_type get_area() const { return (y2 - y1) * (x2 - x1); }

		/**
		 *Checks to see if the rectangle is backwards.
		 *
		 *@returns True if the rectangle's upper-left is above and left (or equal to) the rectangle's lower-right, false if it is not.
		 */
		bool is_valid() const {	return x2 >= x1 && y2 >= y1; }
		
		/**
		 *Shifts the rectangle by a given amount.
		 *
		 *@param pos The amount to shift the rectangle by.
		 */
		rectangle& operator+=( const position& pos ) { ul += pos; lr += pos; return (*this); }

		/**
		 *Returns a rectangle shifted by a given amount.
		 *
		 *@param pos The amount to shift by. 
		 *@returns The shifted rectangle.
		 */
		rectangle operator+( const position& pos ) const {	rectangle r(*this); return r += pos; }

		/**
		 *Shifts the rectangle by a given amount.
		 *
		 *@param pos The amount to shift the rectangle by.
		 */
		rectangle& operator-=( const position& pos ) { ul -= pos; lr -= pos; return (*this); }

		/**
		 *Returns a rectangle shifted by a given amount.
		 *
		 *@oaram pos The amount to shift by.
		 *@returns The shifted rectangle.
		 */
		rectangle operator-( const position& pos ) const {	rectangle r(*this); return r -= pos; }

		/**
		 *Compares two rectangles to see if they are the same.
		 *
		 *@param r The rectangle to compare to.
		 *@returns True if the rectangles have the same positions and dimensions, false otherwise.
		 */
		bool operator==( const rectangle& r ) const { return r.ul == ul && r.lr == lr; }

		/**
		 *Compares two rectangles to see if they are different.
		 *
		 *@param r The rectangle to compare to.
		 *@returns True if the rectangles have different positions or dimensions, false otherwise.
		 */
		bool operator!=( const rectangle& r ) const { return r.ul != ul || r.lr != lr; }

		/**
		 *Checks if a position is within the bounds of this rectangle.
		 *
		 *@param r The position to check.
		 *@returns True if the position is inside or on the edge of the rectangle, false otherwise.
		 */
		bool contains( const position& r ) const{ return y2 >= r.y && y1 <= r.y && x2 >= r.x && x1 <= r.x; }

		/**
		 *Checks if a rectangle is within the bounds of this rectangle.
		 *
		 *@param r The rectangle to check.
		 *@returns True if the entire rectangle to check is inside or on the edge of this rectangle, false otherwise.
		 */
		bool contains( const rectangle& r ) const { return contains( r.ul ) && contains( r.lr ); }

		/**
		 *Checks if another rectangle overlaps this one.
		 *
		 *@param r The rectangle to check.
		 *@returns True if any part of the rectangle to check overlaps this rectangle, false otherwise.
		 */
		bool collides( const rectangle& r ) const { return y2 > r.y1 && y1 < r.y2 && x2 > r.x1 && x1 < r.x2; }

		/**
		 *Limits the region of the rectangle to the inidicated rectangle.
		 *
		 *@param r The rectangle that this rectangle is confined to.
		 *@returns True if the rectangle was changed, false otherwise.
		 */
		bool clamp_to( const rectangle& r )
		{
			if (r.contains(*this)) return false;
			ul = glm::max( ul, r.ul );
			lr = glm::min( lr, r.lr );
			ul = glm::min( ul, lr );
			return true;
		}

		/**
		 *Limits the size of the rectangle to the given rectangle's size.
		 *
		 *@param r The rectangle representing the maximum dimensions this rectangle can be.
		 *@returns True if the rectangle needed to be resized, false otherwise.
		 */
		bool constrain_to( const rectangle& r )
		{
			if ( r.get_width() < get_width() || r.get_height() < get_height() ) return false;
			(*this) += glm::min( r.ul - ul, position() );
			(*this) -= glm::min( lr - r.lr, position() );
			return true;
		}

		/**
		 *Fixes an invalid rectangle.
		 */
		void repair() 
		{
			if (x1 > x2) std::swap( x1, x2 );
			if (y1 > y2) std::swap( y1, y2 );
		}

		/**
		 *Expands the rectangle to just include the given point.
		 *
		 *@param p The point to include in the rectangle.
		 */
		void include_point( position p )
		{
			lr = glm::max( lr, p );
			ul = glm::min( ul, p );
		}

	};

}

#endif NV_POSITION_HH
