// Copyright (C) 2012-2013 ChaosForge / Kornel Kisielewicz
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh
/**
 * @file time.hh
 * @author Kornel Kisielewicz epyon@chaosforge.org
 * @brief Time related functions
 */

#ifndef NV_TIME_HH
#define NV_TIME_HH

#include <nv/common.hh>

namespace nv
{
	/**
	 * Returns the amount of ticks of the processors high resolution 
	 * timer. Currently only supported on GCC and MSVC. Probably only on
	 * some architectures.
	 *
	 * @returns amount of ticks
	 */
	volatile uint64 get_ticks();

	/**
	 * Performs an operating system sleep call.
	 *
	 * @param time time in milliseconds to sleep
	 */
	void sleep( uint32 ms );

	/**
	 * Get millisecond count based on std::clock
	 */
	uint32 get_cpu_ms();

	/**
	 * Get microsecond count based on std::clock
	 */
	uint32 get_cpu_us();

	/**
	 * Get millisecond count based on system counter
	 */
	uint32 get_system_ms();

	/**
	 * Get microsecond count based on system counter
	 */
	uint32 get_system_us();

	struct cpu_ms_timer { uint32 operator()() { return get_cpu_ms(); } };
	struct cpu_us_timer { uint32 operator()() { return get_cpu_us(); } };
	struct system_ms_timer { uint32 operator()() { return get_system_ms(); } };
	struct system_us_timer { uint32 operator()() { return get_system_us(); } };
	
	/**
	 * Timer template class
	 */
	template < class Timer >
	class timer_class
	{
	public:
		timer_class()	: last( Timer()() ) {}
		void mark()	
		{
			uint32 now = Timer()();
			stored = now - last;
			last = now;
		}
		uint32 elapsed() const
		{
			return stored;
		}
	private:
		uint32 last;
		uint32 stored;
	};

	/**
	 * FPS template class
	 */
	template < class Timer >
	class fps_counter_class
	{
	public:
		fps_counter_class() : frames(1), last(0) {}
		bool tick()
		{
			uint32 now = Timer()();
			if ( now - last >= 1000 )
			{
				value = (static_cast<float>(frames) /
					static_cast<float>(now - last))*1000;
				frames = 1;
				last = now;
				return true;
			}
			frames++;
			return false;
		}
		f32 fps() const
		{
			return value;
		}
	private:
		uint32 last;
		uint32 frames;
		f32 value;
	};

	typedef timer_class< system_ms_timer > ms_timer;
	typedef timer_class< system_us_timer > us_timer;
	typedef fps_counter_class< system_ms_timer > fps_counter;

} // namespace nv

#endif // NV_TIME_HH
